% -------------------------------------------------------------------------
% MODEL RECONSTRUCTION OF E-COLI AND ODE SOLVER - Continuous Culture
% -------------------------------------------------------------------------

% -----------------------------------------------------------------------
% Author: Ahmad Abdullah Mannan (c)
% Date: April 2014
% Affiliation : Faculty of Health and Medical Sciences, University of
% Surrey, Guildford, GU2 7TE, UK.
% -----------------------------------------------------------------------

function [xdot flux] = EColi_StableKM_ContCult(t,x_values) %#ok<INUSL>

% -------------------------------------------------------------------------
% Reconstruction of the Central Metabolism of E-Coli:
%   - The chemical rxns and stoichiometry were based on genome-scale model
%     of Fiest and Palsson et al, 2007 paper
%   - Kinetic rxn eqs and parameters have been adopted from literature
%   - Equations have been adjusted to account for regulation as from BRENDA
%     and EcoCyc databases
% -------------------------------------------------------------------------
% 
%	[t,x] = ode15s(@EColi_StableKM_ContCult, [0, t_end], EColi_StableKM_ContCult)
%
% The species in this model are related to the output vectors with the following indices
%   Index	Species name
%	  1  	  X - Biomass
%	  2  	  glc-D
%	  3  	  g6p
%	  4  	  g1p
%	  5  	  f6p
%	  6  	  fdp
%	  7  	  dhap
%	  8  	  g3p
%	  9  	  13dpg
%	  10  	  3pg
%	  11  	  2pg
%	  12  	  pep
%	  13  	  pyr
%	  14  	  6pgc
%	  15	  ru5p-D
%	  16  	  xu5p-D
%	  17  	  r5p
%	  18      s7p
%     19      e4p
%     20      accoa
%     21      actp
%     22      ac - Extracellular
%     23      oaa
%     24      cit
%     25      icit
%     26      akg
%     27      glx
%     28      succ
%     29      succoa
%     30      fum
%     31      mal-L
%             coa
%             pi
%             atp
%             adp
%             amp
%             nadph
%             nadp
%             nadh
%             nad
%             aconC
%             6pgl


%---------------------------------------------------------
% Output Vector:
%---------------------------------------------------------
xdot = zeros(31, 1);

% --------------------------------------------------------
% Calling Parameter Values:
% --------------------------------------------------------
% ParamAdj; %#ok<VUNUS>
EColi_Parameters_StableKM
mu_GSM = 0.1997702967133552607476;
NetFlux_ConnxRxns = [0; % X
                     0; % glcD
                     1.42006406633755E-11; % g6p
                     -3.86535248253495E-12; % g1p
                     -0.018872599662914; % f6p
                     -2.28972396598692E-12; % fdp
                     -0.0560027059350428; % dhap
                     0.00760227878095066; % g3p
                     1.13686837721616E-13; % 13dpg
                     -0.351277110316801; % 3pg
                     9.64343826126424E-06; % 2pg
                     -0.151873881791281; % pep
                     -0.564693444669169; % pyr
                     -9.2192919964873E-13; % 6pgc
                     -0.00795085780916921; % ru5pD
                     -3.33066907387547E-16; % xu5pD
                     -0.18448486874422; % r5p
                     0; % s7p
                     -0.0693179320641093; % e4p
                     -1.19680217901464; % accoa
                     1.13686837721616E-13; % actp
                     0.0901975223880527; % ac
                     -0.575645688450466; % oaa
                     -2.22910578884239E-12; % cit
                     0; % icit
                     -0.226902563480391; % akg
                     0; % glx
                     0.0456434332061466; % succ
                     -0.045064431354265; % succoa
                     0.191804515550393; % fum
                     -6.42784994574774E-12]; % mal

% Concentrations that we consider as fixed constants only:
    coa = 0.26508; % From Keio Multi-Omics Database: Units in mM_c.
    pi = 1.13283938089286; % Found from optimization: Units in mM_c.
    atp = 1.23216; % From Keio Multi-Omics Database: Units in mM_c.
    adp = 0.56225; % From Keio Multi-Omics Database: Units in mM_c.
    amp = 0.57059; % From Keio Multi-Omics Database: Units in mM_c.
    nadph = 0.16293; % From Keio Multi-Omics Database: Units in mM_c.
    nadp = 0.08734; % From Keio Multi-Omics Database: Units in mM_c.
    nadh = 0.00272128861446139; % Found from optimization: Units in mM_c.
    nad = 0.72018; % From Keio Multi-Omics Database: Units in mM_c.
    % aconC = 1;
    % m_6pgl = 1;

%--------------------------------------------------------
% Initial Species Values - these may be overridden by assignment rules
% Note: L_v = litres of culture volume; L_c = litres of single cell volume.

if (nargin == 0)
	% Initial concentrations if not already specified:
    X = 1.56157; %(gDCW/L_v)
    glcD = 0.00885; % From Keio Multi-Omics Database: Units in mM_v.
    g6p = 0.18097; % From Keio Multi-Omics Database: Units in mM_c.
    g1p = 0.06355; % From Keio Multi-Omics Database: Units in mM_c.
    f6p = 0.04389; % From Keio Multi-Omics Database: Units in mM_c.
    fdp = 0.02593; % From Keio Multi-Omics Database: Units in mM_c.
    dhap = 0.12623; % From Keio Multi-Omics Database: Units in mM_c.
    g3p = 0.00274301723549643; % Found from optimization: Units in mM_c.
    m_13dpg = 0.278135900920656; % Found from optimization: Units in mM_c.
    m_3pg = 0.50626; % From Keio Multi-Omics Database: Units in mM_c.
    m_2pg = 0.499302393726613; % Found from optimization: Units in mM_c.
    pep = 0.12009; % From Keio Multi-Omics Database: Units in mM_c.
    pyr = 0.12820; % From Keio Multi-Omics Database: Units in mM_c.
    m_6pgc = 0.0423518003898276; % Found from optimization: Units in mM_c.
    ru5pD = 0.127695; % From Keio Multi-Omics Database: Units in mM_c.
    xu5pD = 0.130200276546662; % Found from optimization: Units in mM_c.
    r5p = 0.03624; % From Keio Multi-Omics Database: Units in mM_c.
    s7p = 0.22159; % From Keio Multi-Omics Database: Units in mM_c.
    e4p = 0.00293513260337421;%0.00293513260337421; % Found from optimization: Units in mM_c.
    accoa = 0.19290; % From Keio Multi-Omics Database: Units in mM_c.
    actp = 0.0435068183291382; % Found from optimization: Units in mM_c.
    ac = 0.00351; % From Keio Multi-Omics Database: Units in mM_v.
    oaa = 0.0321389503914609; % Found from optimization: Units in mM_c.
    cit = 1; % Found from optimization: Units in mM_c.
    icit = 0.00352474324530654; % Found from optimization: Units in mM_c.
    akg = 0.03006; % From Keio Multi-Omics Database: Units in mM_c.
    glx = 0.0906046694779454; % Found from optimization: Units in mM_c.
    succ = 0.06952; % From Keio Multi-Omics Database: Units in mM_c.
    succoa = 0.00936; % From Keio Multi-Omics Database: Units in mM_c.
    fum = 0.05711; % From Keio Multi-Omics Database: Units in mM_c.
    mal = 0.06865; % From Keio Multi-Omics Database: Units in mM_c.
    
    t = 0; %#ok<NASGU> % Initial time point
    
else
	% Floating Species Concentrations
    X = x_values(1);
    glcD = x_values(2);
  	g6p = x_values(3);
  	g1p = x_values(4);
  	f6p = x_values(5);
  	fdp = x_values(6);
  	dhap = x_values(7);
  	g3p = x_values(8);
  	m_13dpg = x_values(9);
  	m_3pg = x_values(10);
  	m_2pg = x_values(11);
  	pep = x_values(12);
  	pyr = x_values(13);
  	m_6pgc = x_values(14);
  	ru5pD = x_values(15);
  	xu5pD = x_values(16);
  	r5p = x_values(17);
    s7p = x_values(18);
    e4p = x_values(19);
    accoa = x_values(20);
    actp = x_values(21);
    ac = x_values(22);
    oaa = x_values(23);
    cit = x_values(24);
    icit = x_values(25);
    akg = x_values(26);
    glx = x_values(27);
    succ = x_values(28);
    succoa = x_values(29);
    fum = x_values(30);
    mal = x_values(31);
    
    t = 0; %#ok<NASGU>
end;

% --------------------------------------------------------
% calculate concentration values

if (nargin == 0)

	% initial concentrations
    xdot(1) = X;
	xdot(2) = glcD;
	xdot(3) = g6p;
	xdot(4) = g1p;
	xdot(5) = f6p;
	xdot(6) = fdp;
	xdot(7) = dhap;
	xdot(8) = g3p;
	xdot(9) = m_13dpg;
	xdot(10) = m_3pg;
	xdot(11) = m_2pg;
	xdot(12) = pep;
 	xdot(13) = pyr;
    xdot(14) = m_6pgc;
    xdot(15) = ru5pD;
    xdot(16) = xu5pD;
    xdot(17) = r5p;
    xdot(18) = s7p;
    xdot(19) = e4p;
    xdot(20) = accoa;
    xdot(21) = actp;
    xdot(22) = ac;
    xdot(23) = oaa;
    xdot(24) = cit;
    xdot(25) = icit;
    xdot(26) = akg;
    xdot(27) = glx;
    xdot(28) = succ;
    xdot(29) = succoa;
    xdot(30) = fum;
    xdot(31) = mal;
    
else


% --------------------------------------------------------
% Reaction Rate/Flux Equations:
% --------------------------------------------------------
% -- Glycolysis ----------------------
    % Phosphotransferase System (PTS):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Irreversible birreactant mechanism, with
        %            non-competitive product inhibition by g6p.
        % Regulation Source: Chassagnole et al (2002).
        %v_PTS = (r1_v_PTS_max * glcD * (pep/pyr)) / ((r1_K_a1 + (r1_K_a2*(pep/pyr)) + (r1_K_a3*glcD) + (glcD*(pep/pyr))) * (1 + ((g6p^r1_n)/r1_K_g6p)));
        if glcD > 0
            v_PTS = (r1_v_PTS_max * glcD * pep) / (((r1_K_a1*pyr) + (r1_K_a2*pep) + (r1_K_a3*glcD*pyr) + (glcD*pep)) * (1 + ((g6p^r1_n)/r1_K_g6p)));
        else
            % If [glcD], [pep] and [pyr] go to zero, eqn contains point of
            % singularity. Hence, impose that if [glcD] = 0 => rate = 0.
            v_PTS = 0; 
        end
        flux(1) = v_PTS;
    % Phosphoglucomutase (PGMT):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Reversible M-M, with accoa, succoa and coa inhibiting
        % uncompetitively against g1p
        % Regulation Source: Duckworth et al (1973)
        NonCompInh = 1 / ( 1 + (accoa/r2_Kiaccoa) + (succoa/r2_Kisuccoa) + (coa/r2_Kicoa) );
        v_PGMT = ( r2_v_PGMT_max * (g6p - ((g1p/r2_K_eq)*NonCompInh)) ) / ( (r2_K_g6p*(1 + (g1p/r2_K_g1p))) + g6p );
        flux(2) = v_PGMT;
    % Phosphoglucose Isomerase (PGI):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Reversible reaction with competitive inhibition from
        %      6pgc. Inhibition constants for 6pgc come from Schreyer and
        %      Bock (1980).
        % Regulation Source: Chassangnole et al (2002)
        v_PGI = ( r3_v_PGI_max * (g6p - (f6p/r3_K_eq)) ) / ( (r3_K_g6p * (1 + (f6p / (r3_K_f6p * (1 + (m_6pgc/r3_K_6pginh_f6p)))) + (m_6pgc/r3_K_6pginh_g6p))) + g6p );
        flux(3) = v_PGI;
    % Phosphofructokinase (PFK):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Monod-Wyman-Changeux kinetics, with allosteric
        %     regulation. Various constants taken from non-optimized values
        %     as in Chassagnole et al (2002).
        % Regulation Source: Chassagnole et al (2002)
        A = 1 + (pep/r4_K_pep) + (adp/r4_K_adp_b) + (amp/r4_K_amp_b);
        B = 1 + (adp/r4_K_adp_a) + (amp/r4_K_amp_a);
        v_PFK = (r4_v_PFK_max * atp * f6p) / ( (atp + (r4_K_atp_s*(1+(adp/r4_K_adp_c)))) * (f6p + (r4_K_f6p_s*A/B)) * (1 + (r4_L/((1 + f6p*B/(r4_K_f6p_s*A))^r4_n)))  );
        flux(4) = v_PFK;
    % Fructose Bisphosphate Aldolase (FBA):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Ordered Uni-Bi mechanism. From EcoCyc, there are two
        %     classes of enzymes, classI = fbaB and classII = fbaA. It is
        %     understood that the class I enzyme is activated by citrate
        %     and pep, however that class of enzymes is known to only be
        %     active during the catabolism of c-3 carbon sources, and so
        %     really only present for gluconeogenesis and not glycolysis.
        %     Since our model is for the catabolism of glucose we will not
        %     consider the class I of enzymes, and so do not need to
        %     account for activation by pep and citrate. Class II of
        %     enzymes is active under glucose metabolism, and has no known
        %     activators. Inhibitor is a non-metabolic compound and will be
        %     neglected for the mechanism.
        % Regulation Source: Chassagnole et al (2002), and EcoCyc for
        % decision of the representation of the enzymatic kinetics of only
        % the class II enzyme, fbaA.
        v_FBA = ( r5_v_FBA_max * (fdp - (g3p*dhap/r5_K_eq)) ) / ( r5_K_fdp + fdp + ((r5_K_g3p*dhap)/(r5_K_eq*r5_v_blf)) + ((r5_K_dhap*g3p)/(r5_K_eq*r5_v_blf)) + ((fdp*g3p)/r5_K_inh_g3p) + (g3p*dhap/(r5_K_eq*r5_v_blf)) );
        flux(5) = v_FBA;
    % Triosephosphate Isomerase (TPI):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Reversible Michaelis-Menten kinetics.
        % Regulation: No known regulation by other metabolites.
        % Regulation Source: Chassagnole et al (2002) and EcoCyc, to
        % confirm that there are no known metabolite regulators of the rxn.
        v_TPI = ( r6_v_TPI_max*(dhap - (g3p/r6_K_eq)) ) / ( (r6_K_dhap*(1 + (g3p/r6_K_g3p))) + dhap);
        flux(6) = v_TPI;
    % Glyceraldehyde 3-Phosphate Dehydrogenase (GAPD):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Random-ordered Bi-Bi Mechanism (also known as a two
        %     substrate reversible Michaelis-Menten). The regulation of the
        %     enzyme is known to be only inhibited by and inorganic
        %     compound arsenate. Since it is not a dynamic variable of our
        %     KM, we will ignore its regulatory effects.
        % Regulation Source: Bakker et al (1997), and EcoCyc for
        % potential but ignored regulator arsenate.
        v_GAPD = ( r7_v_GAPD_max * ((g3p*nad) - (m_13dpg*nadh/r7_K_eq)) ) / ( ((r7_K_g3p*(1 + (m_13dpg/r7_K_pgp))) + g3p) * ((r7_K_nad*(1 + (nadh/r7_K_nadh))) + nad) );
        flux(7) = v_GAPD;
    % Phosphoglycerate Kinase (PGK):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Random-ordered Bi-Bi Mechanism (also can be thought of
        %     as a two substrate reversible Michaelis-Menten). The
        %     preference of the reaction is in the direction of 13dpg
        %     synthesis as atp and 3pg are known activators (EcoCyc).
        % Regulation Source: Chassagnole et al (2002), and EcoCyc for
        % description of preferential flow of direction of rxn flux.
        v_PGK = ( r8_v_PGK_max * ((adp*m_13dpg) - (atp*m_3pg/r8_K_eq)) ) / ( ((r8_K_adp*(1 + (atp/r8_K_atp))) + adp) * ((r8_K_13dpg*(1 + (m_3pg/r8_K_3pg))) + m_13dpg) );
        flux(8) = v_PGK;
    % Phosphoglycerate Mutase (PGM):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Simple reversible Michaelis-Menten kinetics. There are
        %     two enzymes that govern the PGM rxn, GpmA and GpmM. The paper
        %     of Fraser et al (1999) says that the activity of GpmA is 10-
        %     fold higher than that of GpmM. Therefore we decided to ignore
        %     the enzymatic effect of GpmA, and will thus only consider the
        %     metabolic kinetics and regulation of GpmA.
        %     A known regulator of the enzyme is inhibitor vanadate 
        %     (sourced from EcoCyc). However, since this metabolite is not
        %     included in the KM, it's inhibitory effect will be ignored
        %     for our KM. 
        % Regulation Source: Chassagnole et al (2002), and EcoCyc and
        %     Fraser et al (1999) for decision of only considering kinetics
        %     induced by the activity of enzyme GpmA and not GpmM.
        v_PGM = ( (r9_v_PGM_maxf * m_3pg/r9_K_3pg) - (r9_v_PGM_maxr * m_2pg/r9_K_2pg) ) / ( 1 + (m_2pg/r9_K_2pg) + (m_3pg/r9_K_3pg) );
        flux(9) = v_PGM;
    % Enolase (ENO):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: Simple reversible Michaelis-Menten kinetics, with
        %     inhibition from fluoride (sourced from EcoCyc). Since 
        %     fluoride is not included in our KM, it's inhibitory effects
        %     will be ignored for our model.
        % Regulation Source: Chassagnole et al (2002), and EcoCyc for
        %     ignored inhibitor.
        v_ENO = ( r10_v_ENO_max * (m_2pg - (pep/r10_K_eq)) ) / ( (r10_K_2pg*(1 + (pep/r10_K_pep))) + m_2pg );
        flux(10) = v_ENO;
    % Pyruvate Kinase (PYK):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: From EcoCyc we find that the PYK reaction activity is
        % controlled by two co-existing enzymes, each under independent
        % genetic control, PykF (PykI) and PykA (PykII). From EcoCyc: With
        % regards to PykI, it is strongly activated (allosterically) by fdp
        % and shows sigmoidal kinetics with respect to the substrate pep.
        % This behaviour however becomes more hyperbolic on the addition of
        % fdp. The dependency of the reaction behaviour on adp is always
        % hyperbolic. The substrates adp and pep are understood to bind
        % in random order to the enzyme. Inhibitors include succoa and atp.
        % With regards to PykII, this enzyme displays only limited
        % cooperativity amoung pep binding sites and is allosterically
        % activated by amp and several sugar phosphates (EcoCyc).
        % From Chassagnole et al (2002): The reaction mechanism is taken
        % from studies of the pyruvate kinase from yeast, however the
        % regulation is similar to that of E.coli, i.e. Monod-Wyman-
        % -Changeux kinetics of allosteric regulation, with inhibitor atp
        % and activators fdp and amp. Fdp and amp are both allosteric
        % activators of the reaction, however the former is the activator
        % of PykI and the latter is the activator of PykII. Similarly atp
        % is understood to have inhibiting effects on PykI. The following
        % mechanism, as was used in Chassagnole et al (2002) combines the
        % both the regulation of PykI and PykII in a single reaction,
        % assuming that both enzymes are co-existing and that the sum of
        % their concentrations gives a pseudo-complex PykI and II.
        % Regulation Source: All description above of regulation was taken
        % from studies as reported in EcoCyc about PykI and PykII.
        v_PYK = ( r11_v_PYK_max * pep * (((pep/r11_K_pep) + 1)^(r11_n - 1)) * adp ) / ( r11_K_pep * ((r11_L*(((1 + (atp/r11_K_atp))/((fdp/r11_K_fdp) + (amp/r11_K_amp) + 1))^r11_n)) + (((pep/r11_K_pep) + 1)^r11_n)) * (adp + r11_K_adp) );
        flux(11) = v_PYK;
    % Pyruvate Dehydrogenase (PDH):
        % Equation Source: Hoefnagel et al (2002)
        % Regulation: From EcoCyc: Pyruvate dehydrogenase is one of the
        %     most complicated enzyme systems known. The self-assembling
        %     complex is composed of 3 subunits E1(AceE), E2(AceF) and
        %     E3(Lpd) in stoichiometry of 24:24:12, respectively. From
        %     Hoefnagel et al (2002): For the PDH reaction an inhibition by
        %     high NADH:NAD ratio has been described and these kinetics
        %     have been included in the equation. The competitive product
        %     inhibitor accoa to substrate coa and competitive product
        %     inhibitor nadh to nad has also been included into the
        %     reaction kinetics. Furthermore, from Bisswanger et al (1981),
        %     it was found that glyoxylate appears to be a strong
        %     competitive inhibitor with respect to pyruvate, with a Ki
        %     value of 0.5mM, which was found to lie in the range of the
        %     Michaelis constant for pyruvate.
        % Regulation source: Hoefnagel (2002) and Bisswanger (1981)
        v_PDH = ( r12_v_PDH_max * (1/(1 + (r12_Ki*nadh/nad))) * (pyr/r12_K_pyr) * (nad/r12_K_nad) * (coa/r12_K_coa) ) / ( (1 + (pyr/r12_K_pyr) + (glx/r12_Ki_glx)) * (1 + (nad/r12_K_nad) + (nadh/r12_K_nadh)) * (1 + (coa/r12_K_coa) + (accoa/r12_K_accoa)) );
        flux(12) = v_PDH;

% -- Acetate Production/Consumption --
    % Phosphotransacetylase (PTAr):
        % Equation Source: Campos-Bermudez et al (2010) and Wang et al
        %         (2001) for mechanism of regulators.
        % Mechanism: Forward and reverse Hill reaction kinetics. In reverse
        %         rxn: Assumed independant non-competitive inhibition by
        %         nadh, atp, pep and pyr. In forward rxn: Assumed
        %         independant non-competitive inhibition by nadh and atp,
        %         with pep and pyr assumed to be acting independantly as
        %         non-essential activators. The assumed independance means
        %         that the non-essential terms of each activator multiply
        %         the overall reaction equation.
        % Regulation Source: Wang et al (2001), and assumptions, as below:
        % Note: The metabolites regulating the rxns do so allosterically.
        %     It cannot be inferred from the paper exactly the mechanism of
        %     regulation by each of the regulators. It is thus assumed that
        %     each of the regulators acts independant of one another, hence
        %     the multiplicative terms of each regulatory metabolite.
        PTAr_fwd = ( r13_vmax_f * ((accoa/r13_K_accoa)^r13_Haccoa) * (pi/r13_K_pi) ) / ( 1 + ((accoa/r13_K_accoa)^r13_Haccoa) + (pi/r13_K_pi) + (((accoa/r13_K_accoa)^r13_Haccoa)*(pi/r13_K_pi)) );
        PTAr_rev = ( r13_vmax_r * ((coa/r13_K_coa)^r13_Hcoa) * (actp/r13_K_actp) ) / ( 1 + ((coa/r13_K_coa)^r13_Hcoa) + (actp/r13_K_actp) + (((coa/r13_K_coa)^r13_Hcoa)*(actp/r13_K_actp)) );
        v_PTAr = ( PTAr_fwd * (1/(1 + (nadh/r13_Ki_nadhf) + (atp/r13_Ki_atpf))) * ((1 + (r13_v_pep*pep/r13_Ka_pep))/(1 + (pep/r13_Ka_pep))) * ((1 + (r13_v_pyr*pyr/r13_Ka_pyr))/(1 + (pyr/r13_Ka_pyr))) ) - ( PTAr_rev / (1 + (nadh/r13_Ki_nadh) + (atp/r13_Ki_atp) + (pep/r13_Ki_pep) + (pyr/r13_Ki_pyr)) );
        flux(13) = v_PTAr;
    % Acetate Kinase (ACKr):
        % Equation Source: The form of the equation was taken from Segal
        %    (1975) Enzyme Kinetics book, though the mechanism was taken
        %    from Janson et al (1974).
        % Mechanism: Sequential Random-order Mechanism for forward and
        %    reverse reaction directions, with actp showing non-competitive
        %    product inhibition to the forward reaction only, versus both
        %    acetate and ATP.
        % Note: Reaction is: atp + ac <==> adp + actp, which is in opposite
        %    direction to expected rxn, but this is corrected by '(-1)*'
        %    term in v_ACKr equation below.
        % Regulation Source: Specific activities taken from Fox et al
        %    (1985), but otherwise the regulation was understood from
        %    Janson et al (1974).
        v_ACKr_fwd = ( r14_vmaxf * (atp*ac/(r14_alpha*r14_Katp*r14_Kac)) ) / ( 1 + (atp/r14_Katp) + (ac/r14_Kac) + (actp/r14_Kiactp) + (atp*ac/(r14_alpha*r14_Katp*r14_Kac)) + (atp*actp/(r14_Katp*r14_Kiactp)) + (ac*actp/(r14_Kac*r14_Kiactp)) + (atp*ac*actp/(r14_alpha*r14_Katp*r14_Kac*r14_Kiactp)) );
        v_ACKr_rev = ( r14_vmaxr * adp * actp ) / ( (r14_Kiadp*r14_Kactp) + (r14_Kadp*actp) + (r14_Kactp*adp) + (adp*actp) );
        v_ACKr = v_ACKr_fwd - v_ACKr_rev;
        v_ACKr = (-1) * v_ACKr;
        flux(14) = v_ACKr;
    % Acetyl-CoA Synthetase (ACS):
        % Equation Source: Brown (1977)
        % Mechanism: Simple irreversible Michaelis-Menten kinetics,
        % assuming independent binding sites for each substrate, i.e.
        % randome order binding with a ternirary complex.
        % Regulation Source: Unknown regulation of ACS by other mets.
        % v_ACS = ( r15_v_ACS_max * ac * coa * atp ) / ( (r15_K_ac + ac) * (r15_K_coa + coa) * (r15_K_atp + atp) );
        v_ACS = 0;
        flux(15) = v_ACS;

% -- TCA and Glyoxylate Cycle --------
    % Citrate Synthase (CS):
        % Equation Source: Mogilerskaya (2009, Chapter 10)
        % Mechanism: From EcoCyc: Citrate synthase is a typeII enzyme,
        % which appears to be unique to gram-negative bacteria. It is
        % understood to behave like a trimer of dimeric subunits: the dimer
        % is the basic catalytic subunit but the hexameric form is required
        % for allosteric inhibition by NADH. This is not the case for
        % citrate synthase from animals and plants, and even some other
        % bacteria. From Mogilevskaya et al (2009, chapter 10): The enzyme
        % has been understood to be under complex regulation by some key
        % metabolites such as atp, nadh and akg. There has been some
        % disagreement over the exact mechanism of binding which occurs
        % with the enzyme; whether it binds in a random or ordered manner,
        % and if ordered it is not know in which order. For the mechanism
        % that they propose, they assume that accoa binds first (according
        % to atp studies by Jangaard et al 1968). Citrate synthase is known
        % to have a complex regulatory pattern, where the activity of the
        % enzyme is pH-dependent and is modulated by inhibitors atp, nadh
        % and akg. In modelling the kinetics of citrate synthase the
        % following matters have been considered:
        %   [1] - The reaction rate will not show sigmoidal behaviour, as
        %      this is only seen for concentration of KCl > 0.1mM. Since
        %      this concentration is physiological for E.coli, sigmoidicity
        %      will be ignored.
        %   [2] - The reaction is irreversible, consistent with that
        %      assumed in the genome-scale model.
        %   [3] - NADH and akg inhibit CS non-competitively with oaa.
        %   [4] - ATP inhibits CS competitive to accoa, and non-competitive
        %      to oaa.
        %   [5] - The enzyme's maximal activity depends on pH (without
        %      inhibitors) and to the addition of ATP. ATP was shown by
        %      Jangaard et al (1968) to move the bell-shaped pH dependence
        %      to the right and decrease the rate at the maximum point.
        %   [6] - We assume a fixed pH value of 7.0.
        % MECHANISM - Using experimental studies of Jangaard et al (1968),
        %      Mogilevskaya et al (2009) understood that atp inhibited
        %      competitive to accoa and non-competitive to oaa, which could
        %      only happen if accoa is the first substrate. Hence they
        %      assume that the kinetic mechanism of citrate synthase
        %      functions according to an IRREVERSIBLE ORDERED BI-BI
        %      MECHANSIM, WITH ACCOA BINDING FIRST, AND THE INCLUSION OF
        %      THE DEPENDENCY OF ENZYME ACTIVITY ON pH.
        %      (For the reaction scheme please see Mogilevskaya et al
        %      (2009, chapter 10).
        % Regulation Source: Molilevskaya et al (2009) used the
        %      experimental studies of Jangaard et al (1968) to include
        %      regulation into the enzyme kinetic equation.
        pH = 7;
        H = 10^((-1)*pH);
        v_CS = ( r16_v_CS_max * (1+(r16_K_Hd1/H)+(H/r16_K_Hd2)) * accoa * oaa ) / ( (((r16_K_daccoa*r16_K_moaa) + (r16_K_maccoa*oaa))*(1 + (r16_K_Hd1/H) + ((H/r16_K_Hd2)*(1 + (atp/r16_K_iatp))))) + (accoa * r16_K_moaa * (1 + (r16_K_Hd1/H) + (H/r16_K_Hd2) + (akg/r16_K_i1akg) + (nadh/r16_K_i1nadh))) + (accoa * oaa * (1 + (r16_K_Hd1/H) + (H/r16_K_Hd2) + (akg/r16_K_i2akg) + (nadh/r16_K_i2nadh))) );
        flux(16) = v_CS;
    % AconitaseB (ACONTb):
        % Equation Source: Tsuchiya et al (2009)
        % Important Note: Justification for only considering AcnB dimer:
        %     There are two aconitases, acnA and acnB [Jordan (1999)]:
        %     - AcnA: Aerobic-stationary-phase enzyme, induced by iron and
        %             oxidative stress.
        %     - AcnB: Major TCA cycle enzyme: Keio experiments done in
        %             these conds, i.e. aerobic, not in oxidative-stress.
        % We therefore only consider modelling the enzyme activity of the
        % AcnB enzyme, where the subunit composition of AcnB is [AcnB]*2
        % (i.e. a dimeric enzyme).
        % Mechanism: Enzyme kinetic studies of Tsuchiya et al (2009)
        %     clearly showed sigmoidal curves of enzyme activity indicating
        %     co-operative effects => REVERSIBLE HILL EQUATION.
        % Regulation Source: From EcoCyc: There is no known inhibitory or
        %     activatory effects by any other metabolite. Hence the
        %     mechanism is a simple Hill equation, as reported in Tsuchiya
        %     et al (2009).
        v_ACONTb = ( (r17_v_ACONTb_max_f * (cit^r17_nf)) / ((r17_K_df^r17_nf) + (cit^r17_nf)) ) - ( (r17_v_ACONTb_max_r * (icit^r17_nr)) / ((r17_K_dr^r17_nr) + (icit^r17_nr)) );
        flux(17) = v_ACONTb;
    % AconitaseA (ACONTa):
        % We assume that the reactions ACONTa rapidly equilibriates with
        % changes to reaction rate of ACONTb, i.e. that of the two ACONTb
        % is rate limiting and that the concentration of the intermediate
        % metabolite aconitate is constant. Thus we are assuming that the
        % rxn rate of ACONTa is exactly the same as ACONTb (i.e. we
        % essentially lump the two rxns together).
        % Even though the intermediate metabolite is also understood to be
        % a substrate of another reaction in the genome-scale model, namely
        % ACONIs, when performing FVA on the re-parameterized genome-scale
        % model of E.coli we find that the flux range and flux value of the
        % rxn ACONIs is zero. Hence the intermediate metabolite aconitate
        % is only used in the two rxns ACONTa and ACONTb from which we
        % assume that the metabolite concentration remains as steady state.
        v_ACONTa = v_ACONTb;
        flux(36) = v_ACONTa;
    % Isocitrate Dehydrogenase (ICDHyr):
        % Equation Source: Ogawa et al (2007)
        %    [Another reaction mechanism and equation proposed in Russian
        %    paper of Mogilevskaya et al (2007), included the
        %    phosphorylation and de-phosphorylation of the ICDH enzyme. Did
        %    not base the following equation and mechanism based on this
        %    paper for purposes of keeping it simple.]
        % Mechanism: From EcoCyc: Isocitrate dehydrogenase is the first
        %    bacterial enzyme shown to be regulated by phosphorylation and
        %    dephosphorylation. Phosphorylation of the enzyme on a serine
        %    residue by ICDH kinase/phosphatase inactivates it, and
        %    dephosphorylation by the phosphatase reactivates it. The
        %    modulation of this key enzyme activity enables E.coli to make
        %    rapid shifts between the TCA cycle and glyoxylate bypass
        %    pathways. EcoCyc also reports that the enzyme is competitively
        %    inhibited by glx and oaa, which was sources from Nimmo (1986).
        %    - From Nimmo (1986): From the inhibition studies the reaction
        %      mechanism of the ICDH reaction was elucidated. The paper
        %      explains that the shape of the inhibition curves in fact
        %      explain that the enzyme obeys a compulsory-order mechanism
        %      with the co-enzyme NADP binding first.
        %    - EcoCyc: Though the reaction may be considered as reversible,
        %      it is physiologically favoured in the direction of akg
        %      synthesis. Thus we will assume that the reaction is
        %      uni-directional.
        %    - With regards to regulation:
        %      - From Ogawa et al (2007): The enzyme has been shown to be
        %        allosterically inhibited by pep.
        %      - From Nimmo (1986): From the inhibition studies of glx and
        %        oaa it was found that neither glyoxylate nor oaa alone, at
        %        concentration of 0.5mM (relatively high to physiological
        %        conditions within the cell), affected activity of ICDH.
        %        However, inhibition was observed when both compounds were
        %        present in the assay. When compounds were added
        %        sequentially to 0.15mM to an assay of ICDH (independent of
        %        order), a slow increase in inhibition of ICDH activity was
        %        observed. It was discussed the the inhibition studies
        %        showed that the inhibition by the two compounds was
        %        competitive with respect to icit and uncompetitive with
        %        respect to NADP. However, the paper discusses that the
        %        intracellular concentration of glx and oaa growing on
        %        glucose is probably very low. If the concentration of icit
        %        was 0.5mM and the concentrations of glx and oaa 0.06 ICDH
        %        activity would only be inhibited some 3%. This clearly
        %        suggests that the inhibition is not physiologically
        %        significant. Hence we will assume that there is no
        %        inhibition by glx or oaa.
        %    => IRREVERSIBLE ORDERED BI-TER Mechanism, WITH NADP BINDING
        %       FIRST AND ICIT BINDING SECOND AND WITH ALLOSTERIC
        %       INHIBITION BY PEP, INHIBITING ICIT.
        % Regulation Source: Nimmo (1986) to clarify the main reaction
        %      equation mechanism; Ogawa et al (2007) to include the
        %      allosteric inhibition by pep; and Segal (1975) for the
        %      equation forms for the main reaction mechanism and for the
        %      equation for allosteric inhibition by pep.
        % Extra Notes for the calculation of kinetic parameters:
        %      The paper of Ogawa et al (2007), though it gives the
        %      following kinetic parameters: Km of icit = 0.029mM and Ki of
        %      pep = 0.31mM, it does not however give the Km of nadp or the
        %      vmax value. Hence, since we understood that the reaction
        %      mechanism in the absense of the inhibitor is an irreversible
        %      ordered bi-ter mechanism, we can find Km of nadp and vmax by
        %      fitting the known form of the equation of the mechanism to
        %      the experimental data, as presented in Fig.1A of Ogawa et al
        %      (2007). The fitting was done using a constrained
        %      optimization function of MATLAB, fmincon, to find values for
        %      vmax and Knadp whilst minimizing the distance of equation
        %      curve to the experimental data of Ogawa et al (2007).
        ICDH_AllostInh = ( (1 + (icit/r18_Km_icit))^r18_n ) / ( (r18_L*((1 + (pep/r18_Ki_pep))^r18_n)) + ((1 + (icit/r18_Km_icit))^r18_n) );
        v_ICDH = ( r18_v_ICDH_max * (icit / r18_Km_icit) * (nadp / r18_Km_nadp) * ICDH_AllostInh) / ( 1 + (nadp/r18_Km_nadp) + ((icit/r18_Km_icit)*(nadp/r18_Km_nadp)) );
        flux(18) = v_ICDH;
    % Isocitrate Lyase (ICL):
        % Equation Source: Mackintosh and Nimmo (1988), and Segal (Book,
        %     1975).
        % WARNING: Mechanism as proposed by Mackintosh et al is different
        %          and is based on a reversable icl reaction. Also see
        %          Ogawa et al (2007) for the analysis of the
        %          non-competitive inhibition by pep on ICL.
        % Mechanism: Assumed to be irreversible M-M, with mixed
        %      inhibitions, where pep is non-competitive (NC) with icit,
        %      3pg is competitive (C), and akg and succ are both NC.
        % Assumptions: Assuming that only one competitive inhibitor can bind at once
        %              Assuming that non-competitive inhibitors compete
        %              amongst themselves (only bind one at a time)
        % v_ICL = ( (r19_v_ICL_max * (icit/r19_K_icit)) / (1 + (pi/r19_Ki_pi) + (m_3pg/r19_Ki_3pg) + (icit/r19_K_icit)) ) * ( 1 / (1 + (succ/r19_Ki_succ) + (pep/r19_Ki_pep) + (akg/r19_Ki_akg)) );
        % Since the reaction is believed to be inactive during the aerobic
        % conditions, as specified in the genome-scale model (Feist et al
        % 2007) by the zero flux constraints on the ICL reaction, we
        % therefore also decide to switch off the reaction for the KM.
        v_ICL = 0;
        flux(19) = v_ICL;
    % 2-Oxogluterate Dehydrogenase (AKGDH):
        % Equation Source: Wright et al (1992) and Gupta et al (1980 - for
        %     mechanism).
        %     Alternative source: Linda et al (1977), but for pig heart
        % Mechanism: From EcoCyc: This rxn is governed by a large enzyme
        %     complex, where its subunit composition is [(SucA)*12] +
        %     [(SucB)*24] + [(Lpd)*2]. Though the rxn can be thought of as
        %     reversible, the physiological rxn direction is in favour of
        %     flux in the positive direction, i.e. in the direction of
        %     succoa synthesis. It is not reported in EcoCyc what the
        %     enzyme inhibitors or activators could be.
        %     From Wright et al (1992): In comparing metabolite and enzyme
        %     concentrations in several organisms, metabolite concs were
        %     found to be more consistent. They argue that even the kinetic
        %     constants, Km and Ki values, are relatively consistent and
        %     often similar for a given enzyme isolated from different
        %     organisms. They give examples of concentrations and kinetic
        %     parameter values from seven different organisms to show that
        %     they are of the same order of magnitude, and even similar in
        %     value in some of the cases. This consistency among different
        %     organisms, they argue, suggested that these values were
        %     reliable as input data for metabolic models. We thus assume
        %     the same and treat any kinetic parameter values from Wright
        %     et al (1992) as being similar to those of E.coli.
        %     Wright et al (1992) give the reaction mechanism of AKGDH
        %     complex as a MULTISITE PING-PONG WITH PRODUCT INHIBITION.
        %     From Gupta et al (1980): Inhibition studies of glx with the
        %     purified E.coli AKGDH complex clearly show non-competitive
        %     inhibition of glx to the enzyme activity. The Ki value is
        %     reported to be 1.7mM.
        % Overall mechanism:
        %     MULTISITE PING-PONG, WITH PRODUCT INHIBITION AND
        %     NON-COMPETITIVE INHIBITION BY GLX.
        % WARNING: This mechanism is take from organism Dictyostelium
        %     discoideum.
        v_AKGDH = (( r20_v_AKGDH_max * akg * coa * nad ) / ( (r20_K_nad*akg*coa) + (r20_K_coa*akg*nad) + (r20_K_akg*coa*nad) + (akg*coa*nad) + (r20_K_akg*r20_K_z*succoa*nadh/r20_Ki_succoa) + (r20_K_nad*akg*coa*nadh/r20_Ki_nadh) + (r20_K_coa*akg*nad*succoa/r20_Ki_succoa) + (r20_K_akg*r20_K_z*akg*succoa*nadh/(r20_Ki_akg*r20_Ki_succoa)) )) * (1 / (1 + (glx/r20_Ki_glx)));
        flux(20) = v_AKGDH;
    % Succinyl-Coa Synthetase (SUCOAS):
        % Equation Source: Boyer (The Enzymes, Vol.10), Moffet et al (1970)
        %     and Segal (1975, book pgs 710-711)
        % Mechanism: From EcoCyc: The E.coli enzyme are like tetramers.
        %     Genes for the two subunits, SucD and SucC, are coordinately
        %     expressed, so that they are produced in equimolar
        %     proportions. The rxn, in general is considered reversible.
        %     This is the only reaction of the TCA cycle that employs
        %     substrate level phosphorylation. The substrate passes through
        %     the phosphorylated enzyme intermediate.
        %     From Moffet et al (1970): SuccinylCoA synthetase from
        %     E.coli does not exhibit steady state kinetic patterns
        %     indicative of ping-pong kinetics. Instead, kinetic patterns
        %     are consistent with the sequential addition of all substrates
        %     to form a quarternary complex of the enzyme and 3 substrates
        %     before the release of products. The reaction in the direction
        %     of succoa synthesis follows a Rapid Equilibrium Hybrid
        %     Random-Ordered Terreactant System (Ordered A, Random B and C,
        %     where A = ATP, B = coa and C = succ). Please see Moffet et al
        %     (1970) for the reaction schematic and reaction equation.
        %     From Moffet et al (1970): The kinetic patter cannot be
        %     reconciled with a simple ping-pong mechanism. Furthermore,
        %     preliminary experiments are incompatible with the exclusive
        %     operation of either a simple ping-pong mechanism or of a
        %     sequential mechanism in which atp and adp are the first
        %     substrate to add and the last product to be released, resp.
        %     We will further assume that it's the same mechanism of in the
        %     forward direction, i.e. of succ synthesis, where the kinetic
        %     parameter are taken from Boyer (The Enzymes, Vol 10).
        % Regulation Sources: The schematics and elucidation of the rxn
        %     mechanism comes from Moffet et al (1970); the form of the
        %     rxn equation from Segal (1975); the kinetic parameters in the
        %     direction of succoa synthesis from Moffet et al (1970) and
        %     the kinetic parameter in the direction of succ synthesis from
        %     Boyer (The Enzymes, Vol.10).
        % The general mechanism = REVERSIBLE RAPID EQUILIBRIUM HYBRID
        %     RANDOM-ORDERED TERREACTANT SYSTEM (ORDERED A AND RANDOM B AND
        %     C): For fwd direction A = ADP, B = pi and C = succoa; for
        %     reverse direction A = atp, B = coa and C = succ.
        v_SUCOAS_fwd = ( r21_v_SUCOAS_f * adp*succoa*pi ) / ( (r21_K_adp*r21_K_succoa*r21_K_pi) + (adp*r21_K_succoa*r21_K_pi) + (adp*succoa*r21_K_pi) + (adp*r21_K_succoa*pi) + (r21_K_adp*succoa*pi) + (adp*succoa*pi) );
        v_SUCOAS_rev = ( r21_v_SUCOAS_r * atp*coa*succ ) / ( (r21_K_atp*r21_K_coa*r21_K_succ) + (atp*r21_K_coa*r21_K_succ) + (atp*coa*r21_K_succ) + (atp*r21_K_coa*succ) + (r21_K_atp*coa*succ) + (atp*coa*succ) );
        v_SUCOAS = v_SUCOAS_fwd - v_SUCOAS_rev;
        flux(21) = v_SUCOAS;
    % Succinate Dehydrogenase (SUCDi):
        % Equation Source: Hirsch et al (1963)
        % Assumptions: Kinetics under aerobic conditions, quinone source 
        %    is assumed to be freely available in cell (not rate limiting
        %    thus no term in eqn). Assume, under aerobic conds that reverse
        %    rxn has zero flux (Hirsch 1963).
        % Mechanism: From EcoCyc: The succinate dehydrogenase enzyme is a
        %    large complex of 4 subunits. This enzyme complex has a
        %    subunit composition as follows: [SdhA] + [SdhB] + [SdhC] +
        %    [SdhD]. This enzyme complex also plays an important role in
        %    cellular metabolism and directly connects the TCA cycle with
        %    the respiratory electron transport chain. As part of the TCA
        %    cycle succinate is oxidized to fumerate by the enzyme complex
        %    and electrons are transferred to the membrane quinone pool for
        %    entry into the electron transport chain. E.coli SdhCDAB is a
        %    membrane bound heterotetramer, with the hydrophilic subunits
        %    SdhA and SdhB attached to the cytoplasmic surface of the
        %    plasma membrane via interaction with two hydrophobic integral
        %    membrane subunits SdhC and SdhD. Hence, only the activity of
        %    the subunits SdhC and SdhD are considered as they are the only
        %    subunits to interact with intracellular metabolites.
        %    From Hirsch et al (1963): Previously it was believed that the
        %       complex activates the reaction in the direction of
        %       succinate oxidation and in the reverse direction of
        %       fumerate reduction. However the paper shows that there are
        %       two distinct enzymes catalyzing the fwd and reverse
        %       reactions, however both enzymes can catalyze both
        %       reactions.
        %    Here we will assume an irreversible reaction, as consistent
        %    with the genome-scale model.
        % Final Mechanism: SIMPLE IRREVERSIBLE MICHAELIS-MENTEN.
        % Regulation Sources: Hirsch et al (1963) and EcoCyc, where no
        %    known inhibition or activation is described.
        v_SUCDi = ( r22_v_SD_maxf * succ / r22_K_succ ) / ( 1 + (succ/r22_K_succ) );
        flux(22) = v_SUCDi;
    % Fumerase (FUM):
        % Equation Source: Segal (1975)
        % Mechanism: From EcoCyc: This enzyme is not a complex but has
        %     three isozymes, FumA, FumB and FumC. The cell adapts to
        %     changing environmental oxygen conditions by utilizing the
        %     different isozymes. FumA is a major active enzyme under
        %     microaerophilic conditions (1-2% oxygen), and synthesized
        %     under fermentation and aerobic growth conditions. FumB has
        %     higher affinity for malate than fumarate (reverse direction),
        %     so functions as an alternative enzyme during anaerobiosis. It
        %     also has some activity under microaerophilic conditions. Both
        %     FumA and FumB decrease activity when oxygen levels are >15%.
        %     FumB is 79% similar to FumA protein. FumC has low activity
        %     during anaerobic growth, but is highly active under aerobic
        %     growth (21%). FumC is dissimilar to FumA and FumB. This
        %     provides the cell with an active fumarase under highly
        %     oxidative conditions (>4%), a condition where FumA is
        %     inactivated. Furthermore, from the studies of Ueda et al
        %     (1990), two types of fumerases were found in aerobically
        %     grown E.coli cells, FumA and FumC. FumC is Fe-independent
        %     whose activity is very stable, like that of mammalian cells.
        %     FumA is Fe-dependent whose activity, though it was lost
        %     quickly, could be restored under anaerobic incubation.
        %     The isozyme subunits are: [FumA]*2 or [FumB]*2 or [FumC]*4.
        %     Assumptions: FumC is utilized most under aerobic conditions,
        %     thus this eqn is modelling FumC activity, not FumA or FumB.
        %   ==> SIMPLE REVERSIBLE MICHAELIS-MENTEN MECHANISM:
        % Regulation Source: EcoCyc for Km values and decision of only
        %     considering FumC activity; Ueda et al (1990) for specific
        %     activity in the reverse direction.
        v_FUM = ( (r23_v_FUM_maxf * fum / r23_K_fum) - (r23_v_FUM_maxr * mal / r23_K_mal) ) / ( 1 + (mal/r23_K_mal) + (fum/r23_K_fum) );
        flux(23) = v_FUM;
    % Malate Dehydrogenase (MDH):
        % Equation Source: Segal (1975, book)
        % Mechanism: From EcoCyc: MDH is comprises a dimer of MDH monomers.
        %     A subunit composition of MDH is given by [Mdh]*2. MDH
        %     activity is depressed during anaerobic conditions, with a
        %     2-fold negative change. MDH expression increases during
        %     growth on pyruvate and is low during growth on glucose. There
        %     seem to be no reported inhibitors or activators in either
        %     EcoCyc or BRENDA.
        %     Muslin et al (1995): The paper studies nad-linked malate
        %     dehydrogenase from E.coli. We therefore assume an ordered
        %     binding mechanism, similar to that reported in Wright et al
        %     (1992) for the organism D.discoideum.
        %  ==> REVERSIBLE ORDERED BI-BI MECHANISM, ASSUMING THAT EITHER NAD
        %      OR NADH BINDS FIRST ONTO THE ENZYME, DEPENDING ON THE
        %      DIRECTION OF RXN.
        % Regulation Source: Muslin et al (1995) and Wright et al (1992).
        v_MDH = ( ( r24_v_MDH_f*(nad/r24_Km_nad)*(mal/r24_Km_mal) ) / ( 1 + (nad/r24_Km_nad) + (nad*mal/(r24_Km_nad*r24_Km_mal)) ) ) - ( ( r24_v_MDH_r*(nadh/r24_Km_nadh)*(oaa/r24_Km_oaa) ) / ( 1 + (nadh/r24_Km_nadh) + (nadh*oaa/(r24_Km_nadh*r24_Km_oaa)) ) );
        flux(24) = v_MDH;
        
% -- Anaplorotic Reactions -----------
    % Phosphoenolpyruvate Carboxylase (PPC):
        % Equation Source: Segal (1975, book):
        %     Warning! Activators are included in a form I had created and
        %     not from book.
        % Mechanism: From EcoCyc: Overexpression of Ppc improves the growth
        %     yield on glucose, and increases the production of succinate
        %     from glucose by fermentation. This enzyme is a reulatory
        %     enzyme, which replenishes oaa in the TCA cycle. It is a
        %     unique allosteric enzyme that has many kinds of effectors,
        %     and distinct binding sites for each. The overall structure of
        %     E.coli PPC is a tetramer of four identical subunits, where
        %     the subunit composition is [Ppc]*4, also known as a 'dimer of
        %     dimers'.
        %     From Kai et al (1999): The paper discusses the
        %     crystallization and structure analysis of E.coli Ppc enzyme.
        %     The active site of the main metabolic reaction is assigned to
        %     the C-terminal side. This was found from the identification
        %     of essential residues at this active site by doing
        %     site-directed mutagenesis and chemical modifications at this
        %     region.
        %     From Izui et al (1983): PPC is a unique allosteric enzyme in
        %     that it has many kinds of effectors: it is activated by accoa
        %     fdp, and some other including some long-chain fatty acids,
        %     and inhibited by L-aspartate and L-malate (L-malate
        %     inhibition is taken from Yoshinaga et al (1977)). Such
        %     activation or inhibition is brought about through variation
        %     of the half-saturation concentration of pep rather than
        %     through variation of maximal velocity of the enzyme. For
        %     every compound which causes significant inhibition, the
        %     inhibition was confirmed to be of competitive type according
        %     to the following conventional criterion: the maximal reaction
        %     velocities (vmax) obtained from double reciprocal plots of
        %     pep concentration Vs velocity are not affected by inhibitors.
        % [PLEASE SEE NOTES AS WRITTEN ON PAPER, FOR GREATER DETAILS AND
        % DISCUSSIONS.]
        % Mechanism: HILL RXN EQUATION WITH ALLOSTERIC NON-ESSENTIAL
        %     ACTIVATORS ACCOA AND FDP, AND INHIBITOR MALATE, COMPETITIVE
        %     WITH PEP (ONLY AFFECTING Km VALUES AND NOT V_max).
        r25_v_max_app = r25_v_PPC_max * (1 + (r25_a*accoa/r25_Ka_accoa) + (r25_b*fdp/r25_Ka_fdp) + (r25_a*accoa*r25_b*fdp*r25_e/(r25_Ka_accoa*r25_Ka_fdp)) ) / ( 1 + (accoa/r25_Ka_accoa) + (fdp/r25_Ka_fdp) + (accoa*fdp/(r25_Ka_accoa*r25_Ka_fdp)) ); % To account for non-essential activation by accoa and fdp to PPC rxn, as from Izui et al (1981).
        r25_Kpep_app1 = r25_Km_pep * (1 + (accoa/(r25_c*r25_Ka_accoa)) + (fdp/(r25_d*r25_Ka_fdp)) + (accoa*fdp*r25_e/(r25_c*r25_Ka_accoa*r25_d*r25_Ka_fdp)) ) / ( 1 + (accoa/r25_Ka_accoa) + (fdp/r25_Ka_fdp) + (accoa*fdp/(r25_Ka_accoa*r25_Ka_fdp)) ); % To account for non-essential activation by accoa and fdp to PPC rxn, as from Izui et al (1981).
        r25_Kpep_app2 = r25_Kpep_app1 * ( 1 + (mal/r25_Ki_mal) ); % Accounting for competitive inhibition of malate Vs pep, as from Izui et al (1983).
        v_PPC = r25_v_max_app * (pep^r25_n1) / ((r25_Kpep_app2^r25_n1) + (pep^r25_n1));
        flux(25) = v_PPC;
    % Phosphoenolpyruvate Carboxykinase (PPCK):
        % Equation Source: Yang et al (2003)
        % Mechanism: From EcoCyc: PPCK is unusual in that it appears to be
        %     a monomeric enzyme with an allosteric site on its surface.
        %     Though this reaction considered as being reversible, it is
        %     physiologically favoured in the direction of pep synthesis,
        %     hence it will be considered irreversible. Although earlier
        %     studies reported that the enzyme was inhibited by nadh, this
        %     was found to be due to contamination of the partially
        %     purified enzyme preparation by malate dehydrogenase and
        %     interference with the assay reaction. EcoCyc reports of only
        %     two inhibitors: atp and pep (product inhibition).
        %     From BRENDA: It's reported that the enzyme is inhibited by
        %     the following inhibitors: fdp, f6p, dhap, 3pg and nadh (we
        %     shall ignore inhibition by nadh, as was reported in EcoCyc).
        %     However, when investigating the original source, it was that
        %     these inhibitors were speculated to play a physiological role
        %     in the regulation of C4 plant photosynthesis. This
        %     speculation was not projected to bacterial metabolism.
        %     Mechanism is therefore taken from Yang et al (2003), which is
        %     consistent with the regulation of PPCK reported in EcoCyc.
        %     From Yang et al (2003): The kinetics of PPCK in E.coli was
        %     assumed to follow the rapid equilibrium mechanism based on
        %     the study of Krebs and Bridger (1980). Calculations of the
        %     v_max, from steady state chemostat experiments at different
        %     dilution rates of 0.1, 0.32 and 0.55 h-1 showed that the
        %     v_max values show no significant changes between different
        %     dilution rates, agreeing with their observations of the
        %     measure of specific activities. Furthermore, it was found
        %     that the in-vivo PPCK fluxes relative to their maximal rate
        %     values were less than 15%. These results indicate that the
        %     in-vivo regulation of the flux through PPCK according to
        %     growth requirements is exerted at the activity level via the
        %     changes in metabolite concentrations. They further show that
        %     the flux through the PPCK reaction is insensitive to the
        %     ratio of concentrations of ATP:ADP (unless [ADP] < 1.0), but
        %     is sensitive to the concentrations of pep and oaa, when the
        %     concentrations of pep < 1mM and oaa < 2mM. Interestingly, it
        %     was found that the intracellular pep and oaa concentration
        %     pool sizes lie within the most sensitive regulatory domain
        %     for PPCK flux. From this, they discuss that it seems clear
        %     that the in-vivo regulation of PPCK flux in E.coli occurs
        %     mainly by the modulation of enzyme activity and by the
        %     changes in pep and oaa concentrations, rather than by the
        %     concentrations ratio changes of ATP:ADP. An important
        %     inference from their study is that the gluconeogenic PPCK is
        %     not just an enzyme that consumes ATP via a futile cycle
        %     without any biological purposes, but plays a key role in
        %     anaplerosis and metabolic regulation during the growth of
        %     E.coli on glucose.
        %  Overall Mechanism: IRREVERSIBLE RANDOM-ORDER RAPID EQUILIBRIUM
        %     MECHANISM, WITH PRODUCT INHIBITION FROM PEP AND SUBSTRATE
        %     INHIBITION FROM ATP.
        % Regulation Source: Yang et al (2003), which sourced mechanism
        %     from paper of Krebs and Bridger (1980).
        v_PPCK = ( r26_v_PPCK_max * oaa * atp ) / ( (r26_Ki_atp*r26_Km_oaa*atp) + (r26_Km_atp*oaa) + (oaa*atp) + (r26_Ki_atp*r26_Km_oaa*pep/r26_Ki_pep) + (r26_Ki_atp*r26_Km_oaa*adp/r26_Ki_adp) + (r26_Ki_atp*r26_Km_oaa*pep*adp/(r26_Km_pep*r26_Ki_adp)) + (r26_Ki_atp*r26_Km_oaa*atp*pep/(r26_Ki_pep*r26_KI_atp)) + (r26_Ki_atp*r26_Km_oaa*oaa*adp/(r26_Ki_adp*r26_KI_oaa)) );
        flux(26) = v_PPCK;
    % Malic Enzyme (ME1):
        % Equation Source: Wang et al (2007)
        % Mechanism: From EcoCyc: E.coli K-12 encodes two malic enzymes,
        %     which catalyze the decarboxylation of malate to form
        %     pyruvate. One enzyme, MaeA, (modelled here) is nad dependent
        %     and the other, MaeB, is nadp dependent. Though the reaction
        %     is given as reversible, it is understood to be
        %     physiologically favourable in the forward direction, i.e. pep
        %     synthesis. Furthermore, it is reported in EcoCyc that the
        %     reverse reaction, reductive carboxylation of pyruvate to form
        %     malate, is the thermodynamically favoured direction; however,
        %     the kinetic parameters of the enzyme favour the
        %     dehydrogenation/decarboxylation direction. It is reported
        %     that the enzyme has a number of inhibitors: accoa and coa
        %     inhibit allosterically; malate and nad in an uncompetitive
        %     manner; and oaa and atp inhibit, but it is not known how.
        %     EcoCyc reports that the primary regulators of ME1 are accoa
        %     and coa.
        %     From Wang et al (2007): Data conforming to a random
        %     sequential initial velocity pattern (actual kinetic
        %     parameters) were fitted to the following equation:
        %     v = Vmax*A*B/(Kia*Kb + Ka*B + Kb*A + AB). Data conforming to
        %     a rapid equilibrium ordered initial velocity pattern
        %     (apparent kinetic parameters) were fitted to:
        %     v = Vmax*A/(Ka + A). In investigating the kinetic parameters
        %     it was assumed that the total concentration (mg protein/ml)
        %     of extracted protein was fixed.
        %     Substrate inhibition was seen at high concentrations of both
        %     malate and nad. The Km values of nadh-ME1 against malate and
        %     nad were calculated by the Lineweaver-Burk primary plot.
        %     [PLEASE SEE WRITTEN DETAILS OF THE MECHANISM AS IT WAS
        %     DISCOVERED THROUGH VARIOUS PAPERS AND STUDIES.]
        % Mechanism Modeled Here:
        %     IRREVERSIBLE SEQUENTIAL TWO-SUBSTRATE MECHANISM, WITH MWC
        %     ALLOSTERIC INHIBITION BY CoA, WHERE ALLOSTERIC REGULATION IS
        %     ACTING AS A 'K-SYSTEM' REGULATOR.
        % Regulation Source: Wang et al (2006) and Yamaguchi et al (1974).
        %     The form of the equation for basic mechanism was taken from
        %     Wang et al (2006), but the equation for the part of the
        %     allosteric inhibition term is taken from Segal (1975).
        ME1_Inh = ((1 + (mal/r27_K_mal))^r27_n2) / ( (r27_L*((1 + (coa/r27_Ki))^r27_n2)) +  ((1 + (mal/r27_K_mal))^r27_n2) );
        v_ME1 = ME1_Inh * ( r27_v_ME1_max * nad * (mal^r27_n1) ) / ( (r27_K_inad*(r27_K_mal^r27_n1)) + (r27_K_nad*(mal^r27_n1)) + ((r27_K_mal^r27_n1)*nad) + (nad*(mal^r27_n1)) );
        flux(27) = v_ME1;
    % Malate Synthase (MALS):
        % Equation Source: Anstrom (2003)
        % Mechanism: Simple Irreversible Two-subs M-M, with competitive
        % inhibition of pyr with glx
        % Regulation Source: Anstrom (2003)
        % v_MALS = ( r28_v_MALS_max * glx * accoa ) / ( ((r28_Km_glx*(1 + (pyr/r28_Ki_pyr))) + glx) * (r28_Km_accoa + accoa) );
        % ----------------------------------------------------------------
        % Since the reaction is believed to be inactive during the aerobic
        % conditions, as specified in the genome-scale model (Feist et al
        % 2007) by the zero flux constraints on the MALS reaction, we
        % therefore also decide to switch off the reaction for the KM.
        % ----------------------------------------------------------------
        v_MALS = 0;
        flux(28) = v_MALS;
        
% -- Pentose Phosphate Pathway -------
    % Glucose 6-Phosphate Dehydrogenase (G6PDH):
        % Equation Source: Sanwal et al (1970) and Segal (1975)
        % Mechanism: From EcoCyc: The reaction is understood to be
        %     reversible, which is consistent with the thermodynamic
        %     constraints as in the genome-scale model. Studies on the
        %     purified enzyme showed that the enzyme is almost specific for
        %     NADP+ as the coenzyme. With regards to regulation NADPH and
        %     NADH non-competitively inhibit the enzyme activity.
        %     From Sanwal et al (1970): One of the enigmatic problems in
        %     the area of carbohydrate metabolism in bacetria is the nature
        %     of control systems which determine the distribution of
        %     glucose-6-phosphate into the pentose-phosphate pathway and
        %     glycolytic pathway. Previous studies have shown about a
        %     20-30% flux towards the PPP or oxidative pathway in enteric
        %     bacteria. The activity of the crucial enzymes of these
        %     pathways, i.e. G6PDH, GND, TKT and TALA, does not vary
        %     significantly with various growth conditions in E.coli. So,
        %     it seems plausible that the activity of one or more of these
        %     enzymes is modulated by some effectors in the cell.
        %     Alternatively, the ability of the cell to reoxidize the NADPH
        %     (TPNH in older notation) may determine the flow of carbon
        %     through the oxidative pathway. Like some other reactions of
        %     central metabolism, DPNH may also regulate the activity of
        %     one or more of the enzymes in the PPP, a point which could
        %     conceivably be used to explain the fluctuations in the
        %     relative contributions of the oxidative and non-oxidative
        %     pathways under changing conditions.
        %     The purified enzyme is almost specific for NADP+ (TPN+) as
        %     the coenzyme, with < 1/50th of the activity without it.
        %     The reaction is competitively inhibited via product
        %     inhibition by nadph onto nadp. Nadph is also noncompetitively
        %     inhibiting g6p.
        %     Since NADH (DPNH) is understood to be a central control
        %     signal for a large number of enzymes involved in the terminal
        %     oxidative steps of carbohydrate metabolism, the effect of
        %     this compound was tested on G6PDH. From plotted results in
        %     the paper it seemed that the inhibitory effect of NADH caused
        %     a slight cooperative effecct leading to slightly sigmoidal
        %     curves. However, from the double reciprical plots of
        %     substrate concentration Vs velocity, it seems apparent that
        %     the inhibitory effect of NADH is non-competitive with respect
        %     to both G6P and NADP. This is still consistent with the
        %     discussion in the paper which suggests that the inhibitor
        %     binds to an allosteric site, which is very specific to NADH
        %     (even other compounds related to NADH, such as ATP, ADP, AMP,
        %     GTP, AcCoA and CoA, etc where found to be incapable of
        %     inhibition). Therefore, we assume non-competitive inhibition
        %     by NADH, with an assumed Hill-type exponent to account for
        %     the cooperative effect of the inhibition.
        % Important Notes:
        %     [A] Remember that the v_maxs calculated from the experiments
        %     of Sanwal et al (1970) from the data of figures 3 and 7 may
        %     be different due to slightly different assay conditions.
        %     However, this doesn not matter since we should calculate the
        %     v_max of the enzymatic reaction from the known metabolite
        %     concentrations and flux values from Keio and the adjusted
        %     genome-scale model.
        %     [B] Though the reaction is understood to be reversible, we assume
        %     a rapid equilibriation of the formation and consumption of
        %     6pgl => we lump the reactions of G6PDH and PGL together.
        %     Since the reaction of PGL is irreversible, we will therefore
        %     assume that the lumped reaction of G6PDH and PGL is also
        %     irreversible.
        % Final Mechanism:
        %     IRREVERSIBLE BI-TER REACTION, WITH PRODUCT INHIBITION OF
        %     NADPH, COMPETITIVE WITH NADP AND NONCOMPETITIVE TO G6P, WITH
        %     NONCOMPETITIVE INHIBITION BY NADH TO BOTH G6P AND NADP.
        % Regulation Source: Sanwal et al (1970)
        r29_NonCompInh_byNADH = 1 / (1 + (nadh/r29_Ki_nadh)^r29_n);
        r29_CompInh_nadp = (1 + (nadph/r29_Ki_nadph_nadp));
        r29_NonCompInh_g6p = 1 / (1 + (nadph/r29_Ki_nadph_g6p));
        v_G6PDH = r29_v_G6PDH_max*nadp*g6p * r29_NonCompInh_g6p * r29_NonCompInh_byNADH / ( (r29_Ki_nadp*r29_K_g6p*r29_CompInh_nadp) + (r29_K_nadp*r29_CompInh_nadp*g6p) + (nadp*r29_K_g6p) + (nadp*g6p) );
        flux(29) = v_G6PDH;
    % Phosphogluconate Dehydrogenase (GND):
        % Equation Source: DeSilva et al (1979) and Moritz et al (2000)
        % Mechanism: From EcoCyc: Expression of the Gnd enzyme is growth
        %     related. The enzyme is known to be a homodimer, so its
        %     subunit composition is [Gnd]*2. The enzyme is understood to
        %     show high levels of specificity for NADP. Both NADP and ATP
        %     have been understood to inhibit the enzyme activity
        %     competitively with 6pgc (references unaccesable).
        %     From DeSilva et al (1979): The PPP can be thought of as a
        %     biosynthetic pathway leading to r5p, e4p, and s7p, as a
        %     source of reduced NADP+ for biosynthesis or as a cyclic
        %     pathway of carbohydrate metabolism forming g3p.
        %     An experiment done in the paper, with labelled glucose, show
        %     that the oxidative branch of the shunt is used much less, if
        %     at all, anaerobically than aerobically.
        %     Since the enzyme levels do not greatly vary, metabolic
        %     control might determine in-vivo use of the pathway. There are
        %     reports of the E.coli enzyme being inhibited by fdp and atp.
        %     The experimental data in the paper showed that ATP inhibition
        %     curve was sigmoidal with half-maximal inhibition at 3mM, and
        %     the fdp inhibition curve was hyperbolic with half-maximal
        %     inhibition at 0.025mM. This limited data show that E.coli
        %     GND shows fairly complex kinetics and that fdp and ATP are
        %     potential negative effectors in-vivo.
        %     The reaction is understood to have a physiological preference
        %     for a single direction, as opposed to being reversible.
        %     From Moritz et al (2000) [Not for E.coli but for the
        %     bacterial organism Corynebacterium glutamicum]: The
        %     competitive inhibition by NADPH with respect to NADP+ was
        %     consistent with an ordered bi-ter mechanism. However the
        %     order could not be determined (thus we assume that the order
        %     of binding is random). ATP inhibited the GND competitively
        %     with respect to 6pgc. Similarly, GND showed strong inhibition
        %     by FDP, which were found to be inhibiting competitively with
        %     respect to 6pgc.
        %     Even though this analysis was done for the GND enzyme
        %     purified from Corynebacterium glutamicum and not E.coli, we
        %     assume the same basic mechanism, with parameters as found
        %     from investigation of DeSilva et al (1979)!
        % Overall Mechanism:
        %     IRREVERSIBLE BI-TER MECHANISM WITH COMPETITIVE INHIBITION BY
        %     NADPH TO NADP AND COMPETITIVE INHIBITION OF ATP AND FDP WITH
        %     RESPECT TO 6PGC, AS FROM MORITZ ET AL (2000).
        % Regulation Source: DeSilva et al (1979) and Chassagnole et al
        %     (2002).
        v_GND = ( r30_v_GND_max * m_6pgc * nadp ) / ( ( m_6pgc + (r30_K_6pgc*(1+(fdp/r30_Ki_fdp))*(1 + (atp/r30_Ki_atp))) ) * ( nadp + (r30_K_nadp*(1+(nadph/r30_Ki_nadph))) ) );
        flux(30) = v_GND;
    % Ribulose 5-Phosphate 3-Epimerase (RPE):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: From EcoCyc: RPE is an enzyme of the non-oxidative
        %     branch of the pentose phosphate pathway. This reaction is
        %     considered as being reversible.
        %     From Chassagnole et al (2002): Mass Action Kinetics.
        % Regulation Source: None, as regulation unknown. None reported in
        %     EcoCyc or BRENDA.
        v_RPE = r31_v_RPE_max * (ru5pD - (xu5pD/r31_K_eq));
        flux(31) = v_RPE;
    % Ribose 5-Phosphate Isomerase (RPI):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: From EcoCyc: There are two physically and genetically
        %     distinct RPI enzyme isomers present in E.coli, RpiA and RpiB.
        %     RpiA normally account for more than 99% of the enzyme
        %     activity in the cell and functions in the non-oxidative
        %     branch of the pentose phosphate pathway. There is no sequence
        %     similarity between the two isomers of RPI. For the purposes
        %     of our kinetic model, we will only account for the kinetics
        %     as governed by RpiA and not RpiB, since it account for 99% of
        %     RPI activity. The subunit composition of RpiA is [RpiA]*2.
        %     From Chassagnole et al (2002): Mass Action Kinetics.
        % Regulation Source: The rxn seems to be inhibited by AMP, as
        %     Essenberg et al (1975) report that with 2.5mM of AMP there
        %     was about 30% inhibition to the activity of RpiA. However,
        %     since this concentration is very high relative to
        %     physiological concentration values as measured from the cell
        %     (recorded in Keio database), i.e. measured concentrations of
        %     AMP in the cell as growth rates at most 0.7h-1 were about
        %     1.5mM at the most. Therefore, we will ignore the inhibitory
        %     effect of AMP on RpiA, and assume the mass-action kinetic as
        %     given by Chassagnole et al (2002).
        v_RPI = r32_v_RPI_max * (ru5pD - (r5p/r32_K_eq));
        flux(32) = v_RPI;
    % Transketolase A (TKT1):
        % Equation Source: Segal (1975)
        % Mechanism: From EcoCyc: E.coli contains two transketolase
        %     isozymes, TktA and TktB. TktB is responsible for the minor
        %     transketolase activity. TktB expression is increased in
        %     stationary phase. Expression of TktA and TktB is
        %     complementary, resulting in approximately a constant level of
        %     transketolase expression throughout growth.
        %     For simplicity, the enzyme activity and regulation of TktA
        %     have been accounted for in this reaction equation assuming
        %     that TktA plays a larger role in metabolism when the cell is
        %     in exponential phase of growth. TktA enzyme is the reversible
        %     link between glycolysis and the PPP. TktA is responsible for
        %     the major transketolase activity. In addition to its function
        %     in central carbon metabolism, transketolase appears to have
        %     an unexpected role in chromosome structure. The subunit
        %     composition of TktA = [TktA]*2, and that of TktB = [TktB]*1.
        % Final Mechanism: REVERSIBLE MICHAELIS-MENTEN KINETICS
        % Regulation Source: Sprenger et al (1995), where all Km values
        %     were from, but K_eq taken from Chassagnole et al (2002).
        v_TKT1 = r33_v_TKT1_max * ((r5p*xu5pD) - ((g3p*s7p)/r33_K_eq)) / ( ((r33_K_r5p*(1+(g3p/r33_K_g3p))) + r5p) * ((r33_K_xu5pD*(1+(s7p/r33_K_s7p))) + xu5pD) );
        flux(33) = v_TKT1;
    % Transketolase B (TKT2):
        % Equation Source: Segal (1975)
        % Mechanism: Same discussion as for TKT1, as TktA is the catalytic
        %     and regulatory enzyme for both TKT1 and TKT2, as from EcoCyc.
        % Final Mechanism: REVERSIBLE MICHAELIS-MENTEN KINETICS
        % Regulation Source: Sprenger et al (1995), where all Km values
        %     were from, but K_eq taken from Chassagnole et al (2002).
        v_TKT2 = r34_v_TKT2_max * ((e4p*xu5pD) - ((f6p*g3p)/r34_K_eq)) / ( ((r34_K_e4p*(1+(f6p/r34_K_f6p))) + e4p) * ((r34_K_xu5pD*(1+(g3p/r34_K_g3p))) + xu5pD) );
        flux(34) = v_TKT2;
    % Transaldolase (TALA):
        % Equation Source: Chassagnole et al (2002)
        % Mechanism: From EcoCyc: There are two closely related
        %     transaldolases in E.coli, encoded by talA and talB. TalA
        %     enzyme has not been biochemically characterized. Protein
        %     levels of TalA are induced by osmotic stress only under
        %     aerobic conditions. We will therefore only consider the
        %     biochemical kinetic mechanism and regulation of TalB as a
        %     representation of this reaction under aerobic, non-osmotic
        %     stress conditions (assumed).
        %     TalB is an enzyme of the non-oxidative branch of the PPP.
        %     Along with transketolase, transaldolase creates a reversible
        %     link between the PPP and glycolysis. Only TalB has been
        %     biochemically characterized. TalB is a dimer in solution,
        %     where its subunit composition is [TalB]*2.
        %     The reaction is understood to be a reversible reaction.
        %     From Sprenger et al (1995): [PLEASE SEE SHEET OF WRITING FOR
        %     GREATER DETAILS.] 
        % Mechanism: TWO SUBSTRATE REVERSIBLE MICHAELIS-MENTEN KINETICS,
        %     also known as A RANDOM-ORDERED BI-BI MECHANISM.
        v_TALA = ( r35_v_TALA_maxf * ((g3p * s7p) - (e4p * f6p / r35_K_eq)) ) / ( ((r35_K_g3p*(1 + (e4p/r35_K_e4p))) + g3p) * ((r35_K_s7p*(1 + (f6p/r35_K_f6p))) + s7p) );
        flux(35) = v_TALA;
        
    % flux(36) : PLEASE see above (under v_ACONTb, flux17) for v_ACONTa.
    
    % Other extra reactions added in later:
        % 6-phosphogluconolactonase (PGL):
        % Equation Source: None.
        % Mechanism: From EcoCyc: The reaction is understood to be
        %     irreversible in the direction of 6pgc synthesis. This
        %     reaction can also proceed spontaneously, which makes it
        %     difficult to understand the physiological role of the enzyme.
        %     Furthermore, since the metabolite 6pgl[c] is only used in the
        %     two rxns of G6PDH and PGL (as can be seen in the genome-scale
        %     model), and that both are represented in our kinetic model,
        %     and that the rxn is understood to proceed spontaneously, we
        %     will assume a rapid equilibriation of the metabolic pool size
        %     of 6pgl[c]. Therefore we assume that the reaction rate is the
        %     same as that of the reaction rate of G6PHD.
        % Regulation Source: None.
        v_PGL = v_G6PDH;
        flux(37) = v_PGL;


% --------------------------------------------------------
% ORDINARY DIFFERENTIAL EQUATIONS:
% Species Conc. Rate Equations - For Batch Culture
% --------------------------------------------------------
% Conversion factor used for units consistency within ODEs for
% intracellular metabolites: cell density rho_x = CellDryWeight/CellVolume:
rho_X = 2.8e-13/4.96e-16; % Units = g/L_c
    
    % Biomass Production:
        mu = mu_GSM; % Specific growth rate is coming from genome-scale model
        % xdot(1) = mu * X + NetFlux_ConnxRxns(1);
        xdot(1) = (mu - D) * X; % For continuous culture growth.
    % glcD (Extracellular Glucose):
        % xdot(2) = - (v_PTS*X) + NetFlux_ConnxRxns(2);
        %xdot(2) = ((glcD_f - glcD) * D * 0.9696204429543870) - (v_PTS * X); % For continuous culture growth.
        xdot(2) = ((glcD_f - glcD) * D * 0.97073534845439838) - (v_PTS * X); % For continuous culture growth.
    % g6p (Glucose 6 Phosphate):
        xdot(3) = ((v_PTS - v_G6PDH - v_PGMT - v_PGI + NetFlux_ConnxRxns(3)) * rho_X) - ((mu - D) * g6p);
    % g1p (Glucose 1 Phosphate):
        xdot(4) = ((v_PGMT + NetFlux_ConnxRxns(4)) * rho_X) - ((mu - D) * g1p);
    % f6p (Fructose 6 Phosphate):
        xdot(5) = ((v_PGI - v_PFK + v_TKT2 + v_TALA + NetFlux_ConnxRxns(5)) * rho_X) - ((mu - D) * f6p);
    % fdp (Fructose-1,6-Bisphosphate):
        xdot(6) = ((v_PFK - v_FBA + NetFlux_ConnxRxns(6)) * rho_X) - ((mu - D) * fdp);
    % dhap (Dihydroxyacetone Phosphate):
        xdot(7) = ((v_FBA - v_TPI + NetFlux_ConnxRxns(7)) * rho_X) - ((mu - D) * dhap);
    % g3p (Glyceraldehyde 3 Phosphate):
        xdot(8) = ((v_FBA + v_TPI - v_GAPD + v_TKT1 + v_TKT2 - v_TALA + NetFlux_ConnxRxns(8)) * rho_X) - ((mu - D) * g3p);
    % 13dpg (3-Phosphoglycerol Phosphate):
        xdot(9) = ((v_GAPD - v_PGK + NetFlux_ConnxRxns(9)) * rho_X) - ((mu - D) * m_13dpg);
    % 3pg (3-Phosphoglycerate):
        xdot(10) = ((v_PGK - v_PGM + NetFlux_ConnxRxns(10)) * rho_X) - ((mu - D) * m_3pg);
    % 2pg (Glycerate 2-Phosphate):
        xdot(11) = ((v_PGM - v_ENO + NetFlux_ConnxRxns(11)) * rho_X) - ((mu - D) * m_2pg);
    % pep (Phosphoenolpyruvate):
        xdot(12) = ((v_ENO - v_PYK + v_PPCK - v_PPC - v_PTS + NetFlux_ConnxRxns(12)) * rho_X) - ((mu - D) * pep);
    % pyr (Pyruvate):
        xdot(13) = ((v_PTS + v_PYK - v_PDH + v_ME1 + NetFlux_ConnxRxns(13)) * rho_X) - ((mu - D) * pyr);
    % 6pgc (6-Phosphogluconate):
        xdot(14) = ((v_PGL - v_GND + NetFlux_ConnxRxns(14)) * rho_X) - ((mu - D) * m_6pgc);
    % ru5pD (Ribulose 5-Phosphate):
        xdot(15) = ((v_GND - v_RPE - v_RPI + NetFlux_ConnxRxns(15)) * rho_X) - ((mu - D) * ru5pD);
    % xu5pD (Xylulose 5-Phosphate):
        %xdot(15) = v_RPE - v_TKT1 - v_TKT2 - (mu * xu5pD) + NetFlux_ConnxRxns(16);
        xdot(16) = ((v_RPE - v_TKT1 - v_TKT2 + NetFlux_ConnxRxns(16)) * rho_X) - ((mu - D) * xu5pD);
    % r5p (Ribose 5-Phosphate):
        %xdot(16) = v_RPI - v_TKT1 - (mu * r5p) + NetFlux_ConnxRxns(17);
        xdot(17) = ((v_RPI - v_TKT1 + NetFlux_ConnxRxns(17)) * rho_X) - ((mu - D) * r5p);
    % s7p (Sedoheptulose 7-Phosphate):
        xdot(18) = ((v_TKT1 - v_TALA + NetFlux_ConnxRxns(18)) * rho_X) - ((mu - D) * s7p);
    % e4p (Erythrose 4-Phosphate):
        xdot(19) = ((v_TALA - v_TKT2 + NetFlux_ConnxRxns(19)) * rho_X) - ((mu - D) * e4p);
    % accoa (Acetyl-CoA):
        xdot(20) = ((v_PDH - v_CS - v_PTAr + v_ACS - v_MALS + NetFlux_ConnxRxns(20)) * rho_X) - ((mu - D) * accoa);
    % actp (Acetyl Phosphate):
        xdot(21) = ((v_PTAr - v_ACKr + NetFlux_ConnxRxns(21)) * rho_X) - ((mu - D) * actp);
    % ac (Extracellular Acetate):
        %xdot(22) = (v_ACKr - v_ACS)*X - (mu * ac) - (D * ac) + NetFlux_ConnxRxns(21);
        %xdot(22) = (v_ACKr - v_ACS)*X - (mu * ac) + NetFlux_ConnxRxns(22);
        % Treat as intracellular for now:
        xdot(22) = ((v_ACKr - v_ACS + NetFlux_ConnxRxns(22)) * rho_X) - ((mu - D) * ac);
    % oaa (Oxaloacetate):
        xdot(23) = ((v_MDH - v_CS - v_PPCK + v_PPC + NetFlux_ConnxRxns(23)) * rho_X) - ((mu - D) * oaa);
    % cit (Citrate):
        xdot(24) = ((v_CS - v_ACONTa + NetFlux_ConnxRxns(24)) * rho_X) - ((mu - D) * cit);
    % icit (Isocitrate):
        xdot(25) = ((v_ACONTb - v_ICDH - v_ICL + NetFlux_ConnxRxns(25)) * rho_X) - ((mu - D) * icit);
    % akg (Alphaketoglutarate):
        xdot(26) = ((v_ICDH - v_AKGDH + NetFlux_ConnxRxns(26)) * rho_X) - ((mu - D) * akg);
    % glx (Glyoxylate):
        xdot(27) = ((v_ICL - v_MALS + NetFlux_ConnxRxns(27)) * rho_X) - ((mu - D) * glx);
    % succ (Succinate):
        xdot(28) = ((v_ICL + v_SUCOAS - v_SUCDi + NetFlux_ConnxRxns(28)) * rho_X) - ((mu - D) * succ);
    % succoa (Succinyl-CoA):
        xdot(29) = ((v_AKGDH - v_SUCOAS + NetFlux_ConnxRxns(29)) * rho_X) - ((mu - D) * succoa);
    % fum (Fumarate):
        xdot(30) = ((v_SUCDi - v_FUM + NetFlux_ConnxRxns(30)) * rho_X) - ((mu - D) * fum);
    % malL (Malate-L):
        xdot(31) = ((v_FUM + v_MALS - v_MDH - v_ME1 + NetFlux_ConnxRxns(31)) * rho_X) - ((mu - D) * mal);
    
end