% -------------------------------------------------------------------------
% MODEL RECONSTRUCTION OF E-COLI AND ODE SOLVER - Continuous Culture
% -------------------------------------------------------------------------

% -----------------------------------------------------------------------
% Author: Ahmad Abdullah Mannan (c)
% Date: April 2014
% Affiliation : Faculty of Health and Medical Sciences, University of
% Surrey, Guildford, GU2 7TE, UK.
% -----------------------------------------------------------------------

%---------------------------------------------------------
% Calculating Vmax Values from Other Code:
%---------------------------------------------------------
% [vmax] = EColiModel_FindVmaxs_04042012;

% Parameters Determined from Chemostat Experiments:
    D = 0.1997702967133552607476; % Setting dilution rate (units in 'per second')
    glcD_f = (1000.0/180.0) * 4.0; % Glucose concentration in the feed (from g/l to mM)

% --------------------------------------------------------
% Cellular Metabolite Concentrations
% Data Source: Keio University Multi-omics Database
% Data Source Conditions: Wild-type E.coli at dilution = 0.2h-1.
% --------------------------------------------------------
% Units of biomass = gDCW/(litre of culture volume) = gDCW/L_v.
% Units of extracellular metabolites = mmol/(litre of culture volume)
%                                    = mmol/L_v = mM_v.
% Units of intracellular metabolites = mmol/(litre of cell volume)
%                                    = mmol/L_c = mM_c.
% --------------------------------------------------------
    X = 1.56157; %(gDCW/L_v)
    glcD = 0.00885; %(mM) per vol culture - Keio
    g6p = 0.18097;
    g1p = 0.06355;
    f6p = 0.04389;
    fdp = 0.02593;
    dhap = 0.12623;
    g3p = 0.00274301723549643;
    m_13dpg = 0.278135900920656;
    m_3pg = 0.50626;
    m_2pg = 0.499302393726613;
    pep = 0.12009;
    pyr = 0.12820;
    m_6pgc = 0.0423518003898276;
    ru5pD = 0.127695;
    xu5pD = 0.130200276546662;
    r5p = 0.03624;
    s7p = 0.22159;
    e4p = 0.00293513260337421;%0.00293513260337421;
    accoa = 0.19290;
    actp = 0.0435068183291382;
    ac = 0.00351; %(mM) per vol culture - Keio
    oaa = 0.0321389503914609;
    cit = 1;
    icit = 0.00352474324530654;
    akg = 0.03006;
    glx = 0.0906046694779454;
    succ = 0.06952;
    succoa = 0.00936;
    fum = 0.05711;
    mal = 0.06865;
    coa = 0.26508; % From divalent ion of CoA
    pi = 1.13283938089286;
    atp = 1.23216;
    adp = 0.56225;
    amp = 0.57059;
    nadph = 0.16293;
    nadp = 0.08734;
    nadh = 0.00272128861446139;
    nad = 0.72018;

% --------------------------------------------------------
% Reaction Flux Estimates from 13C-Metabolic Flux Analysis
% Data Source: Keio University Multi-omics Database
% --------------------------------------------------------
% Reaction Flux Distribution as Extracted from Adjusted
% Genome-Scale Model of E.coli: iAF1260 GSM parameters
% adjusted to replicate, as close as possible, the 13C-MFA
% data as reported in Keio Multi-Omics Database.
% --------------------------------------------------------
    % 0 Calculated Glucose Uptake Rate:
    % Note: We have a correction factor of 0.9704 to correct the difference
    % in the rPTS flux value between that estimated for the kinetic model
    % and that which is maximally allowed by the genome-scale model. They
    % must be the same for the integration of the two model types to work!
        %rPTS = 0.9696204429543870 * (glcD_f - glcD) * D / X; % Units are in mmol/gDCW/h
        %rPTS = 0.97073534845439838 * (glcD_f - glcD) * D / X; % Units are in mmol/gDCW/h
    % 1 Phosphotransferase System (PTS):
        v_PTS = 2.75857499998358;
    % 2 Phosphoglucomutase (PGMT):
        v_PGMT = 3.86535248253495E-12;
    % 3 Phosphoglucose Isomerase (PGI):
        v_PGI = 2.0433622144177;
    % 4 Phosphofructokinase (PFK):
        v_PFK = 2.34990167674802;
    % 5 Fructose Bisphosphate Aldolase (FBA):
        v_FBA = 2.34990167674573;
    % 6 Triosephosphate Isomerase (TPI):
        v_TPI = 2.29389897081069;
    % 7 Glyceraldehyde 3-Phosphate Dehydrogenase (GAPD):
        v_GAPD = 4.77944999130193;
    % 8 Phosphoglycerate Kinase (PGK):
        v_PGK = 4.77944999130204;
    % 9 Phosphoglycerate Mutase (PGM):
        v_PGM = 4.42817288098524;
    % 10 Enolase (ENO):
        v_ENO = 4.42818252442351;
    % 11 Pyruvate Kinase (PYK):
        v_PYK = 0.907568908109191;
    % 12 Pyruvate Dehydrogenase (PDH):
        v_PDH = 3.10145046342582;
% -- Acetate Production/Consumption --
    % 13 Phosphotransacetylase (PTAr):
        v_PTAr = -0.0901975223881664;
    % 14 Acetate Kinase (ACKr):
        v_ACKr = -0.0901975223880527;
    % 15 Acetyl-CoA Synthetase (ACS):
        v_ACS = 0;
% -- TCA and Glyoxylate Cycle --------
    % 16 Citrate Synthase (CS):
        v_CS = 1.99484580679935;
    % 17 Aconitase (ACONTb):
        v_ACONTb = 1.99484580679712;
    % Isocitrate Dehydrogenase (ICDHyr):
        v_ICDH = 1.99484580679712;
    % Isocitrate Lyase (ICL):
        v_ICL = 0;
    % 2-Oxogluterate Dehydrogenase (AKGDH):
        v_AKGDH = 1.76794324331672;
    % Succinyl-Coa Synthetase (SUCOAS):
        v_SUCOAS = 1.72287881196246;
    % Succinate Dehydrogenase and Fumarate Reductase (SUCDi and FRD2):
        v_SUCDi = 1.76852224516861;
    % Fumerase (FUM):
        v_FUM = 1.960326760719;
    % Malate Dehydrogenase (MDH):
        v_MDH = 1.96032676071036;
% -- Anaplorotic Reactions -----------
    % Phosphoenolpyruvate Carboxylase (PPC):
        v_PPC = 0.610164734541665;
    % Phosphoenolpyruvate Carboxykinase (PPCK):
        v_PPCK = 2.21246405076491E-12;
    % Malic Enzyme (ME1):
        v_ME1 = 2.21234972109508E-12;
    % Malate Synthase (MALS):
        v_MALS = 0;
% -- Pentose Phosphate Pathway -------
    % Glucose 6-Phosphate Dehydrogenase (G6PDH):
        v_G6PDH = 0.715212785576217;
    % Phosphogluconate Dehydrogenase (GND):
        v_GND = 0.715212785575295;
    % Ribulose 5-Phosphate 3-Epimerase (RPE):
        v_RPE = 0.325412061993234;
    % Ribose 5-Phosphate Isomerase (RPI):
        v_RPI = 0.381849865772892;
    % Transketolase A (TKT1):
        v_TKT1 = 0.197364997028671;
    % Transketolase B (TKT2):
        v_TKT2 = 0.128047064964562;
    % Transaldolase (TALA):
        v_TALA = 0.197364997028671;
        

% Setting the adjustment to subset of parameter values to stabilize the
% system:
%ParamAdj = [0.524044427351832,0.635076379010644,0.7693550047951,0.593605860406527,1.09780310847966,0.990122701231135,0.989943576222589,0.993604348989174,1.00714301040913,0.990790169053356,1.00664867812603,0.935502969513304,1.05950876962416,1.00571801085344,0.854906251551476,0.659501188201287,1.01648932829768,0.973742909058362,0.935599074215671,1.00510484205593,1.06158010116265,1.12618885373224,1.26198016873392,1.14263720084853,0.712136817829072,0.5,1.81327169856176,1.27079640275299,0.972351142653679,1.89523576632988,0.970537005579892];
ParamAdj = [0.509703705365313,1.47906316451003,0.854968130479631,0.515844263495389,0.564362556379042,1.04783039019832,0.848862346539814,0.910384353317272,1.1574706196866,1.45847340365063,1.02959669753872,0.666974179472756,0.893340305676735,0.903629338641562,1.25774819134001,1.48713442997534,1.24291390679307,1.44436154104649,1.51885416780364,1.53222210529128,0.637179475592805,1.5799840010884,0.846492081148585,0.970458262450535,1.20489990830213,0.53061460772291,2.25242538077762,0.516743309697166,0.855589747158453,1.53333244841776,1.02517518475496];
adj = ParamAdj;

% --------------------------------------------------------
% Parameter Values:
% --------------------------------------------------------
% Phosphotransferase System (PTS):
    r1_K_a1 = 1.0 * adj(1); % Arbitrary
    r1_K_a2 = 0.01 * adj(2); % Notley-McRobb et al. (1997): 37oC assumed at pH 7.
    r1_K_a3 = 1.0 * adj(3); % Arbitrary
    r1_K_g6p = 0.5 * adj(4); % Kaback (1969): Unknown temperature, but at pH 6.6.
    r1_n = 4.0 * adj(5); % Arbitrary
    r1_v_PTS_max = v_PTS / ( (glcD * pep) / (((r1_K_a1*pyr) + (r1_K_a2*pep) + (r1_K_a3*glcD*pyr) + (glcD*pep)) * (1 + ((g6p^r1_n)/r1_K_g6p))) );
% Phosphoglucomutase (PGMT):
    r2_K_eq = 0.142 * adj(6); % Lowry and Passonneau (1969): 25oC at pH 7.4.
    r2_K_g6p = 0.02 * adj(7); % Lowry and Passonneau (1969): 25oC at pH 7.4.
    r2_K_g1p = 0.008 * adj(8); % Lowry and Passonneau (1969): 25oC at pH 7.4.
    r2_Kiaccoa = 0.0917031 * adj(9); % Source: Calculated from Duckworth (1973): 25oC at pH 7.4.
    r2_Kisuccoa = 0.1494253 * adj(10); % Source: Calculated from Duckworth (1973): 25oC at pH 7.4.
    r2_Kicoa = 0.1025358 * adj(11); % Source: Calculated from Duckworth (1973): 25oC at pH 7.4.
    NonCompInh = 1 / ( 1 + (accoa/r2_Kiaccoa) + (succoa/r2_Kisuccoa) + (coa/r2_Kicoa) );
    r2_v_PGMT_max = v_PGMT / (( (g6p - ((g1p/r2_K_eq)*NonCompInh)) ) / ( (r2_K_g6p*(1 + (g1p/r2_K_g1p))) + g6p ));
% Phosphoglucose Isomerase (PGI):
    %r3_v_PGI_max = 94.0296330462549; % As estimated from optimization.
    r3_K_g6p = 2.46; % Takama and Nosoh (1980): 65oC at pH 8.0.
    r3_K_f6p = 0.2; % Dykhuizen and Hartl (1983): 37oC at pH 7.65.
    r3_K_eq = 0.43; % Pettersson (1990): Unknown temperature, but at pH 7.0.
    r3_K_6pginh_g6p = 0.19; % Schreyer and Bock (1980): 28oC at pH 8.0.
    r3_K_6pginh_f6p = 0.19; % Schreyer and Bock (1980): 28oC at pH 8.0.
    r3_v_PGI_max = v_PGI / ((g6p - (f6p/r3_K_eq)) / ( (r3_K_g6p * (1 + (f6p / (r3_K_f6p * (1 + (m_6pgc/r3_K_6pginh_f6p)))) + (m_6pgc/r3_K_6pginh_g6p))) + g6p ));
% Phosphofructokinase (PFK):
    r4_K_pep = 3.26 * adj(12);
    r4_K_adp_b = 0.25 * adj(13); % Rizzi et al (1997): Unknown experimental conditions.
    r4_K_amp_b = 0.1 * adj(14); % Rizzi et al (1997): Unknown experimental conditions.
    r4_K_adp_a = 239 * adj(15); % Rizzi et al (1997): Unknown experimental conditions.
    r4_K_amp_a = 8.74 * adj(16); % Rizzi et al (1997): Unknown experimental conditions.
    r4_K_atp_s = 0.16 * adj(17); % Deville-Bonne et al (1991): 28oC at pH 8.2.
    r4_K_adp_c = 0.36 * adj(18); % Deville-Bonne et al (1991): 28oC at pH 8.2.
    r4_K_f6p_s = 0.14 * adj(19); % Deville-Bonne et al (1991): 28oC at pH 8.2.
    r4_L = 4000000.0 * adj(20); % Diaz Ricci (1999): Unknown experimental conditions.
    r4_n = 4.0 * adj(21); % Diaz Ricci (1999): Unknown experimental conditions.
    A = 1 + (pep/r4_K_pep) + (adp/r4_K_adp_b) + (amp/r4_K_amp_b);
    B = 1 + (adp/r4_K_adp_a) + (amp/r4_K_amp_a);
    r4_v_PFK_max = v_PFK / ( (atp * f6p) / ( (atp + (r4_K_atp_s*(1+(adp/r4_K_adp_c)))) * (f6p + (r4_K_f6p_s*A/B)) * (1 + (r4_L/((1 + f6p*B/(r4_K_f6p_s*A))^r4_n)))  ) );
% (FBA):
    %r5_kcat = 10.5; % From BRENDA
    %r5_EnzConc = 2.43801479830374; % From Keio Multi-Omics Database
    %r5_EnzWeight = 78.294; % From EcoCyc
    %r5_v_FBA_max = r5_kcat * (r5_EnzConc/r5_EnzWeight) * (3600/1000);
    %r5_v_FBA_max = r5_v_FBA_max * 54.3463879918119;
    r5_K_fdp = 0.133 * 1.75248551184301; % Babul et al (1993): 30oC, pH 7.6.
    r5_K_g3p = 0.088 * 1.90363364665193; % Babul et al (1993): 30oC, pH 7.6.
    r5_K_dhap = 0.088 * 1.49485430463724; % Babul et al (1993): 30oC, pH 7.6.
    r5_K_inh_g3p = 0.6 * 1.31782841817371; % Babul et al (1993): 30oC, pH 7.6.
    r5_K_eq = 0.14 * 0.500070550718323; % Babul et al (1993): 30oC, pH 7.6.
    r5_v_blf = 2.0 * 0.500016671794889; % Babul et al (1993): 30oC, pH 7.6.
    r5_v_FBA_max = v_FBA / ((fdp - (g3p*dhap/r5_K_eq)) / ( r5_K_fdp + fdp + ((r5_K_g3p*dhap)/(r5_K_eq*r5_v_blf)) + ((r5_K_dhap*g3p)/(r5_K_eq*r5_v_blf)) + ((fdp*g3p)/r5_K_inh_g3p) + (g3p*dhap/(r5_K_eq*r5_v_blf)) ));
% (TPI):
    %r6_kcat = 9000; % From BRENDA
    %r6_EnzConc = 0.459272926872108; % From Keio Multi-Omics Database
    %r6_EnzWeight = 53.944; % From EcoCyc
    %r6_v_TPI_max = r6_kcat * (r6_EnzConc/r6_EnzWeight) * (3600/1000);
    %r6_v_TPI_max = r6_v_TPI_max * 0.345989466698443;
    r6_K_eq = 0.04 * 1.69023623247386; % Babul et al (1993): 30oC, pH 7.6.
    r6_K_dhap = 2.8 * 1.22191425765639; % Babul et al (1993): 30oC, pH 7.6.
    r6_K_g3p= 0.3 * 1.91130447171052; % Babul et al (1993): 30oC, pH 7.6.
    r6_v_TPI_max = v_TPI / ((dhap - (g3p/r6_K_eq)) / ( (r6_K_dhap*(1 + (g3p/r6_K_g3p))) + dhap));
% (GAPDH):
    %r7_kcat = 268; % From BRENDA
    %r7_EnzConc = 17.5117296074979; % From Keio Multi-Omics Database
    %r7_EnzWeight = 142.128; % From EcoCyc
    %r7_v_GAPD_max = r7_kcat * (r7_EnzConc/r7_EnzWeight) * (3600/1000);
    %r7_v_GAPD_max = r7_v_GAPD_max * 42.6174848161149;
    r7_K_eq = 0.63 * 1.76424589514252; % Pettersson (1990)
    r7_K_g3p = 0.15 * 0.837548956375786; % Bakker et al (1997): 25oC, but unknown pH value.
    r7_K_pgp = 0.1 * 0.529210438718159; % Bakker et al (1997): 25oC, but unknown pH value.
    r7_K_nad = 0.45 * 1.93700406035562; % Bakker et al (1997): 25oC, but unknown pH value.
    r7_K_nadh = 0.02 * 0.801427861478434; % Bakker et al (1997): 25oC, but unknown pH value.
    r7_v_GAPD_max = v_GAPD / (((g3p*nad) - (m_13dpg*nadh/r7_K_eq)) / ( ((r7_K_g3p*(1 + (m_13dpg/r7_K_pgp))) + g3p) * ((r7_K_nad*(1 + (nadh/r7_K_nadh))) + nad) ));
% (PGK):
    %r8_SpecActivity = 480; % From Fifis et al (1978)
    %r8_EnzConc = 0.691292184435414; % From Keio Multi-Omics Database
    %r8_v_PGK_max = r8_SpecActivity * r8_EnzConc * (60/1000);
    %r8_v_PGK_max = r8_v_PGK_max * 1.09182957519222;
    r8_K_eq = 1800 * 0.762822940514809; % Ni and Savageau (1996): Taken from study of human red blood cell, but unknown conditions.
    r8_K_adp = 0.18 * 1.01696840784591; % Molnar and Vas (1993): Value +/- 0.05mM: 20oC at pH 7.5.
    r8_K_atp = 0.24 * 0.604461708468774; % Fifis and Scopes (1978): 37oC at pH 7.0.
    r8_K_13dpg = 0.006 * 1.90819676999579; % Lavoinne et al (1983): Taken from study of PGK enzyme from rat liver.
    r8_K_3pg = 0.17 * 1.81918185535633; % Schmidt et al (1995): Taken from study of PGK enzyme from yeast.
    r8_v_PGK_max = v_PGK / (((adp*m_13dpg) - (atp*m_3pg/r8_K_eq)) / ( ((r8_K_adp*(1 + (atp/r8_K_atp))) + adp) * ((r8_K_13dpg*(1 + (m_3pg/r8_K_3pg))) + m_13dpg) ));
% (PGM):
    %r9_v_PGM_maxf = 45.6662818624095; % As estimated from optimization.
    r9_v_PGM_maxr = 33.6088802772402;  % As estimated from optimization.
    r9_K_3pg = 0.200; % Fraser et al (1999); Units of mM +/- 0.027mM: 37oC at pH 7.0.
    r9_K_2pg = 0.190; % Fraser et al (1999); Units of mM +/- 0.035mM: 37oC at pH 7.0.
    r9_v_PGM_maxf = ( v_PGM + ((r9_v_PGM_maxr * m_2pg/r9_K_2pg)/(1 + (m_2pg/r9_K_2pg) + (m_3pg/r9_K_3pg))) ) / ((m_3pg/r9_K_3pg) / ( 1 + (m_2pg/r9_K_2pg) + (m_3pg/r9_K_3pg) ));
% (ENO):
    % r10_v_ENO_max = vmax{10};
    %r10_v_ENO_max = 6.33125895151748; % As estimated from optimization.
    r10_K_eq = 6.7; % Spring and Wold (1971): 30oC at pH 8.1.
    r10_K_2pg = 0.1; % Spring and Wold (1971): 30oC at pH 8.1.
    r10_K_pep = 0.135; % Duggleby (1994): Unknown Conditions.
    r10_v_ENO_max = v_ENO / ((m_2pg - (pep/r10_K_eq)) / ( (r10_K_2pg*(1 + (pep/r10_K_pep))) + m_2pg ));
% (PYK):
    % r11_v_PYK_max = vmax{11};
    r11_K_pep = 0.31 * adj(22); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_n = 4.0 * adj(23); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_L = 1000 * adj(24); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_K_atp = 22.5 * adj(25); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_K_fdp = 0.19 * adj(26); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_K_amp = 0.2 * adj(27); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_K_adp = 0.26 * adj(28); % Boiteax et al (1983): 25oC at pH 7.0.
    r11_v_PYK_max = v_PYK / (( pep * (((pep/r11_K_pep) + 1)^(r11_n - 1)) * adp ) / ( r11_K_pep * ((r11_L*(((1 + (atp/r11_K_atp))/((fdp/r11_K_fdp) + (amp/r11_K_amp) + 1))^r11_n)) + (((pep/r11_K_pep) + 1)^r11_n)) * (adp + r11_K_adp) ));
% (PDH):
    %r12_SpecActivity = 28.35; % From BRENDA
    %r12_EnzConc = 103.251666275605; % From Keio Multi-Omics Database
    %r12_v_PDH_max = r12_SpecActivity * r12_EnzConc * (60/1000);
    %r12_v_PDH_max = r12_v_PDH_max * 0.556504488033955;
    r12_Ki = 46.4 * 1.51551304167689; % Hoefnagel et al (2002); Taken from Lactococcus lactis bacteria: 30oC at pH 6.5.
    r12_K_pyr = 0.3 * 1.6720844295624; % Bisswanger et al (1981); Units of mM: 25oC at pH 7.6.
    r12_K_nad = 0.4 * 1.87223874577683; % Snoep et al (1992); Taken from bacteria Lactococcus lactis: 37oC at pH 6.0.
    r12_K_coa = 0.014 * 1.24949822069125; % Hoefnagel et al (2002); Taken from Lactococcus lactis bacteria: 30oC at pH 6.5.
    r12_K_nadh = 0.1 * 1.79994441280583; % Hoefnagel et al (2002); Taken from Lactococcus lactis bacteria: 30oC at pH 6.5.
    r12_K_accoa = 0.008 * 1.36665754589554; % Bresters et al (1975); Taken from Azotobacter vinelandii bacteria: 25oC at pH 7.6.
    r12_Ki_glx = 0.5 * 1.36877086723254; % Bisswanger et al (1981): 25oC at pH 7.6.
    r12_v_PDH_max = v_PDH / (( (1/(1 + (r12_Ki*nadh/nad))) * (pyr/r12_K_pyr) * (nad/r12_K_nad) * (coa/r12_K_coa) ) / ( (1 + (pyr/r12_K_pyr) + (glx/r12_Ki_glx)) * (1 + (nad/r12_K_nad) + (nadh/r12_K_nadh)) * (1 + (coa/r12_K_coa) + (accoa/r12_K_accoa)) ));
% (PTAr):
    % Parameters of enzyme for dilution rate of 0.2h-1, as from Keio data:
    % Note: ENZYME KINETIC STUDIES PERFORMED AT CONDITIONS OF 30oC, pH 8.0.
    enzymeConc = 0.720689539642254; % Keio Multi-Omics Database; Units of 'mg-Protein/gDW'.
    enzymeWeight = 463.032; % EcoCyc: Weight of polypeptides; Units of 'g/mmol'.
    %kcat_fwd = 29.6; % Campos-Bermudez et al (2010); Units of '1/s'.
    %r13_vmax_f = kcat_fwd * (enzymeConc / enzymeWeight) * (3600/1000); % The 3600/1000 is to convert vmax to correct units of 'mmol/(gDW*h)'.
    kcat_rev = 227.6; % Campos-Bermudez et al (2010); Units of '1/s'.
    r13_vmax_r = kcat_rev * (enzymeConc / enzymeWeight) * (3600/1000); % The 3600/1000 is to convert vmax to correct units of 'mmol/(gDW*h)'.
    
    % Parameters for the basic forward rxn:
    %r13_vmax_f = r13_vmax_f * 3.54399324447376;
    r13_K_accoa = (44.9/1000) * 1.45843794990705; % Campos-Bermudez et al (2010); Divided by 1000 to convert from uM to mM units.
    r13_K_pi = 2.1 * 0.791807088657325; % Campos-Bermudez et al (2010); Units in mM.
    r13_Haccoa = 1.3 * 1.63125889337585; % Campos-Bermudez et al (2010); Hill coefficient for accoa term.    
    % Parameters for regulation acting on forward rxn:
    r13_Ki_nadhf = 0.0696 * 1.75716833030015; % Calculated from data from Campos-Bermudez et al (2010)
    r13_Ki_atpf = 0.2667 * 1.76971558505525; % Calculated from data from Campos-Bermudez et al (2010)
    r13_Ka_pep = 0.0479 * 1.23474912067872; % Calculated (optimization) from data from Campos-Bermudez et al (2010)
    r13_Ka_pyr = 1.0642 * 1.34575027395761; % Calculated (optimization) from data from Campos-Bermudez et al (2010)
    r13_v_pep = 1.2051 * 1.73601629795672; % Calculated (optimization) from data from Campos-Bermudez et al (2010)
    r13_v_pyr = 1.3965 * 1.84859304011958; % Calculated (optimization) from data from Campos-Bermudez et al (2010)
    % Parameters for the basic reverse rxn:
    r13_vmax_r = r13_vmax_r * 33.6637862036135;
    r13_K_coa = (67.2/1000) * 1.04677015622289; % Campos-Bermudez et al (2010); Divided by 1000 to convert from uM to mM units.
    r13_K_actp = 0.9 * 1.54557122613874; % Campos-Bermudez et al (2010); Units in mM.
    r13_Hcoa = 1.7 * 0.566575545196052; % Campos-Bermudez et al (2010); Hill coefficient for coa term.
    % Parameters for the regulation acting on the reverse rxn:
    r13_Ki_nadh = 0.1091 * 1.51163621664166; % Calculated from data from Campos-Bermudez et al (2010)
    r13_Ki_atp = 0.239 * 1.32851049863929; % Calculated from data from Campos-Bermudez et al (2010)
    r13_Ki_pep = 2.5333 * 1.82833978869848; % Calculated from data from Campos-Bermudez et al (2010)
    r13_Ki_pyr = 36.7241 * 1.05739726324986; % Calculated from data from Campos-Bermudez et al (2010)
    
    PTAr_rev = ( r13_vmax_r * ((coa/r13_K_coa)^r13_Hcoa) * (actp/r13_K_actp) ) / ( 1 + ((coa/r13_K_coa)^r13_Hcoa) + (actp/r13_K_actp) + (((coa/r13_K_coa)^r13_Hcoa)*(actp/r13_K_actp)) );
    PTAr_fwd = ( v_PTAr + ( PTAr_rev / (1 + (nadh/r13_Ki_nadh) + (atp/r13_Ki_atp) + (pep/r13_Ki_pep) + (pyr/r13_Ki_pyr)) ) ) / ( (1/(1 + (nadh/r13_Ki_nadhf) + (atp/r13_Ki_atpf))) * ((1 + (r13_v_pep*pep/r13_Ka_pep))/(1 + (pep/r13_Ka_pep))) * ((1 + (r13_v_pyr*pyr/r13_Ka_pyr))/(1 + (pyr/r13_Ka_pyr))) );
    r13_vmax_f = PTAr_fwd / (( ((accoa/r13_K_accoa)^r13_Haccoa) * (pi/r13_K_pi) ) / ( 1 + ((accoa/r13_K_accoa)^r13_Haccoa) + (pi/r13_K_pi) + (((accoa/r13_K_accoa)^r13_Haccoa)*(pi/r13_K_pi)) ));
% (ACKr):
    % Note: ENZYME KINETIC STUDIES IN THE FOLLOWING PAPERS WITH PERFORMED
    % UNDER THE RESPECTIVE CONDITIONS:
    %    - Fox et al (1985): 21oC at pH 7.4.
    %    - Janson et al (1974): 25oC at pH 7.0.
    r14_EnzymeConc_Keio = 0.966556889306751; % Keio Multi-Omics Database; Units = mg-Protein/gDCW.
    r14_SpecActivity_f = 2000; % Fox et al (1985); Units of umol/min/mg-Protein
    r14_vmaxf = r14_SpecActivity_f * r14_EnzymeConc_Keio * (60/1000); % Converting v_max value to units of mmol/(gDW*h).
    %r14_SpecActivity_r = 2600; % Fox et al (1985); Units of umol/min/mg-Protein
    %r14_vmaxr = r14_SpecActivity_r * r14_EnzymeConc_Keio * (60/1000); % Converting v_max value to units of mmol/(gDW*h).
    r14_vmaxf = r14_vmaxf * 11.5280289398767;
    %r14_vmaxr = r14_vmaxr * 0.129870419289033;
    r14_alpha = (2/35) * 0.560531663067512; % As calculated from K_a/K_ia, from data of Janson et al (1974); Unitless.
    r14_Katp = 0.35 * 1.03324594754719; % As calculated from K_a/alpha, from data of Janson et al (1974); Units if mM.
    r14_Kac = 101.5 * 0.718549072888525; % As calculated from K_b/alpha, from data of Janson et al (1974); Units if mM.
    r14_Kiactp = 0.715 * 1.81232161688916; % As calculated from the average of Kii values of actp on atp and ac, from data of Janson et al (1974); Units of mM.
    r14_Kiadp = (50/1000) * 0.644715037318088; % Janson et al (1974); Units converted to mM.
    r14_Kactp = 0.34 * 1.47455592518959; % Janson et al (1974); Units converted to mM.
    r14_Kadp = (360/1000) * 1.08771766925732; % Janson et al (1974); Units converted to mM.
    
    v_ACKr_fwd = ( r14_vmaxf * (atp*ac/(r14_alpha*r14_Katp*r14_Kac)) ) / ( 1 + (atp/r14_Katp) + (ac/r14_Kac) + (actp/r14_Kiactp) + (atp*ac/(r14_alpha*r14_Katp*r14_Kac)) + (atp*actp/(r14_Katp*r14_Kiactp)) + (ac*actp/(r14_Kac*r14_Kiactp)) + (atp*ac*actp/(r14_alpha*r14_Katp*r14_Kac*r14_Kiactp)) );
    v_ACKr_rev = v_ACKr + v_ACKr_fwd;
    r14_vmaxr = v_ACKr_rev / (( adp * actp ) / ( (r14_Kiadp*r14_Kactp) + (r14_Kadp*actp) + (r14_Kactp*adp) + (adp*actp) ));
% (ACS):
    % No parameters as reaction is set to zero flux.
    % vmax{15} = 0;
% (CS):
    %r16_kcat = 81.0; % Taken from BRENDA from E.coli wild type enzyme.
    %r16_EnzConc = 15.2309085649793; % From Keio Multi-Omics Database.
    %r16_EnzWeight = 288.09; % As taken from the subunit composition of the enzyme and the weight of polypeptide, as reccorded in EcoCyc.
    %r16_v_CS_max = r16_kcat * (r16_EnzConc/r16_EnzWeight) * (3600/1000);
    %r16_v_CS_max = r16_v_CS_max * 0.592241784910182;
    r16_K_Hd1 = (1.0/100000) * 1.11877296545989; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_Hd2 = (2.0/10000) * 1.00967039137331; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_daccoa = 0.1 * 1.14747420581157; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_moaa = 0.04 * 0.965564425433176; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_maccoa = 0.18 * 1.6688299273682; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_iatp = 0.58 * 0.993801873168849; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_i1akg = 0.015 * 0.985948339960787; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_i1nadh = (3.3/10000) * 1.13238003421698; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_i2akg = 0.256 * 0.99264069068511; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    r16_K_i2nadh = (8.4/1000) * 1.00128693184547; % Mogilevskaya et al (2009); Units of mM: Assuming pH 7.0.
    pH = 7; % As reaction equation accounts for kinetic sensitivity to change in pH of media.
    H = 10^((-1)*pH);
    r16_v_CS_max = v_CS / (( (1+(r16_K_Hd1/H)+(H/r16_K_Hd2)) * accoa * oaa ) / ( (((r16_K_daccoa*r16_K_moaa) + (r16_K_maccoa*oaa))*(1 + (r16_K_Hd1/H) + ((H/r16_K_Hd2)*(1 + (atp/r16_K_iatp))))) + (accoa * r16_K_moaa * (1 + (r16_K_Hd1/H) + (H/r16_K_Hd2) + (akg/r16_K_i1akg) + (nadh/r16_K_i1nadh))) + (accoa * oaa * (1 + (r16_K_Hd1/H) + (H/r16_K_Hd2) + (akg/r16_K_i2akg) + (nadh/r16_K_i2nadh))) ));
% (ACONTb):
    %r17_v_ACONTb_max_f = 40.4196129595618; % As estimated from optimization.
    r17_v_ACONTb_max_r = 0; % As estimated from optimization.
    r17_nf = 1.229; % Tsuchiya et al (2009); Value +/- 0.055 : Using data from AcnB: pH 7.5 but Unknown temperature, so assuming 25oC (room temperture).
    r17_K_df = 11.1; % Tsuchiya et al (2009); Value +/- 1.5 : Using data from AcnB: pH 7.5 but Unknown temperature, so assuming 25oC (room temperture).
    r17_nr = 0.727; % Tsuchiya et al (2009); Value +/- 0.021 : Using data from AcnB: pH 7.5 but Unknown temperature, so assuming 25oC (room temperture).
    r17_K_dr = 0.741; % Tsuchiya et al (2009); Value +/- 0.007 : Using data from AcnB: pH 7.5 but Unknown temperature, so assuming 25oC (room temperture).
    r17_v_ACONTb_max_f = (v_ACONTb + ( (r17_v_ACONTb_max_r * (icit^r17_nr)) / ((r17_K_dr^r17_nr) + (icit^r17_nr)) )) / ((cit^r17_nf) / ((r17_K_df^r17_nf) + (cit^r17_nf)));
% (ICDHyr):
    %r18_v_ICDH_max = 28.2329579300234; % As estimated from optimization.
    r18_Km_icit = 0.029; % mM - Ogawa et al (2007): pH 7.1, but unknown temperature.
    r18_Ki_pep = 0.31; % mM - Ogawa et al (2007): pH 7.1.
    r18_Km_nadp = 0.005; % mM - As estimated from Ogawa et al (2007) data, Fig.1A: pH 7.1.
    r18_L = 0.3709; % As estimated from Ogawa et al (2007) data, Fig.1A: pH 7.1.
    r18_n = 1.0048; % As estimated from Ogawa et al (2007) data, Fig.1A: pH 7.1.
    ICDH_AllostInh = ( (1 + (icit/r18_Km_icit))^r18_n ) / ( (r18_L*((1 + (pep/r18_Ki_pep))^r18_n)) + ((1 + (icit/r18_Km_icit))^r18_n) );
    r18_v_ICDH_max = v_ICDH / (( (icit / r18_Km_icit) * (nadp / r18_Km_nadp) * ICDH_AllostInh ) / ( 1 + (nadp/r18_Km_nadp) + ((icit/r18_Km_icit)*(nadp/r18_Km_nadp)) ));
% (ICL):
    r19_v_ICL_max = 0;
    r19_K_icit = 0.063; % Mackintosh et al (1988): 25oC at pH 7.3.
    r19_Ki_pi = 11.0; % Mackintosh et al (1988): 25oC at assumed pH of 7.3.
    r19_Ki_3pg = 0.8; % Mackintosh et al (1988): 25oC at pH 7.3.
    r19_Ki_succ = 1.12; % Mackintosh et al (1988): 25oC at pH 7.3.
    r19_Ki_pep = 0.91; % Mackintosh et al (1988): 25oC at pH 7.3.
    r19_Ki_akg = 1.35; % Mackintosh et al (1988): 25oC at pH 7.3.
% (AKGDH):
    %r20_kcat = 49; % Taken from abstract of Waskiewicz et al (1984)
    %r20_EnzConc = 37.5628418379611; % From Keio Multi-Omics Database.
    %r20_EnzWeight = 2418.36; % From EcoCyc.
    %r20_v_AKGDH_max = r20_kcat * (r20_EnzConc/r20_EnzWeight) * (3600/1000);
    %r20_v_AKGDH_max = r20_v_AKGDH_max * 2.84419254329432;
    r20_K_nad = 0.07 * 0.837799804312185; % Wright et al (1992); Units in mM: Unknown experimental conditions for purified enzymes.
    r20_K_coa = 0.002 * 1.88423187133261; % Wright et al (1992); Units in mM: Unknown experimental conditions for purified enzymes.
    r20_K_akg = 0.1 * 0.909601116533607; % Gupta et al (1980) from E.coli; Units in mM: 25oC at pH 8.0.
    r20_K_z = 1.5 * 0.853433429175836; % Wright et al (1992); Units in mM: Unknown experimental conditions for purified enzymes.
    r20_Ki_succoa = 0.004 * 1.24527809118702; % Wright et al (1992); Units in mM: Unknown experimental conditions for purified enzymes.
    r20_Ki_nadh = 0.018 * 1.76081782461058; % Wright et al (1992); Units in mM: Unknown experimental conditions for purified enzymes.
    r20_Ki_akg = 0.75 * 1.51480337459256; % Wright et al (1992); Units in mM: Unknown experimental conditions for purified enzymes.
    r20_Ki_glx = 1.7 * 1.55543560870878; % Gupta et al (1980) from E.coli; Units in mM: 25oC at pH 8.0.
    r20_v_AKGDH_max = (v_AKGDH * (1 + (glx/r20_Ki_glx))) / (( akg * coa * nad ) / ( (r20_K_nad*akg*coa) + (r20_K_coa*akg*nad) + (r20_K_akg*coa*nad) + (akg*coa*nad) + (r20_K_akg*r20_K_z*succoa*nadh/r20_Ki_succoa) + (r20_K_nad*akg*coa*nadh/r20_Ki_nadh) + (r20_K_coa*akg*nad*succoa/r20_Ki_succoa) + (r20_K_akg*r20_K_z*akg*succoa*nadh/(r20_Ki_akg*r20_Ki_succoa)) ));
% Succinyl-CoA Synthetase (SUCOAS):
    r21_EnzConc = 3.52958762166039; % Keio Multi-Omics database; Units of mg-Protein/gDCW.
    r21_EnzWeight = 142.34; % From EcoCyc; Units of g-Protein/mmol.
    r21_kcat_r = 44.73; % BRENDA Database; Units of s-1.
    %r21_kcat_f = 44.73; % Assuming the same for the forward direction (since papers only seem to report reaction in the direction of succoa synthesis): Units of s-1.
    %r21_v_SUCOAS_f = (r21_EnzConc/r21_EnzWeight) * r21_kcat_f * (3600/1000);
    r21_v_SUCOAS_r = (r21_EnzConc/r21_EnzWeight) * r21_kcat_r * (3600/1000); % Converted to units of mmol/gDW/h.
    %r21_v_SUCOAS_f = r21_v_SUCOAS_f * 7.1624860801555;
    r21_v_SUCOAS_r = r21_v_SUCOAS_r * 0.829392053415815;
    r21_K_adp = 0.012 * 0.936129118698419; % Boyer(The Enzymes Vol.10, Chapter 18, book)
    r21_K_succoa = 0.0077 * 1.37903506325087; % Boyer(The Enzymes Vol.10, Chapter 18, book)
    r21_K_pi = 2.6 * 1.71607210042494; % Boyer(The Enzymes Vol.10, Chapter 18, book)
    r21_K_atp = 0.02 * 1.03876711354984; % Moffet et al (1970)
    r21_K_coa = 0.0015 * 1.03509267750474; % Moffet et al (1970)
    r21_K_succ = 0.1 * 1.67127258756262; % Moffet et al (1970)
    
    v_SUCOAS_rev = ( r21_v_SUCOAS_r * atp*coa*succ ) / ( (r21_K_atp*r21_K_coa*r21_K_succ) + (atp*r21_K_coa*r21_K_succ) + (atp*coa*r21_K_succ) + (atp*r21_K_coa*succ) + (r21_K_atp*coa*succ) + (atp*coa*succ) );
    v_SUCOAS_fwd = v_SUCOAS + v_SUCOAS_rev;
    r21_v_SUCOAS_f = v_SUCOAS_fwd / (( adp*succoa*pi ) / ( (r21_K_adp*r21_K_succoa*r21_K_pi) + (adp*r21_K_succoa*r21_K_pi) + (adp*succoa*r21_K_pi) + (adp*r21_K_succoa*pi) + (r21_K_adp*succoa*pi) + (adp*succoa*pi) ));
% Succinate Dehydrogenase (SUCDi):
    r22_K_succ = 0.26 * adj(29); % Hirsch et al (1963): 30oC at pH 7.7.
    r22_v_SD_maxf = v_SUCDi / (( succ / r22_K_succ ) / ( 1 + (succ/r22_K_succ) ));
% Fumarase (FUM):
    % Sourced papers: Ueda et al (1990) and Woods et al (1988).
    % Woods.SA, Schwartzbach.SD, Guest.JR, 'Two biochemically distinct
    % classes of fumerase in Escherichia coli', Biochem Biophys Acta (1988)
    % Apr28; 954(1):14-26.
    r23_K_fum = 0.39 * adj(30); % BRENDA, sourced from Woods et al (1988); Units of mM (unknown conditions).
    r23_K_mal = 2.94 * adj(31); % BRENDA, sourced from Woods et al (1988); Units of mM (unknown conditions).
    r23_SpecActivity_r = 340; % Ueda et al (1990); Units of umol/min/mg-Protein: 30oC at pH 8.0.
    r23_EnzConc = 1.76777005487691; % Keio Multi-omics database; Units of mg-Protein/gDCW.
    r23_v_FUM_maxr = r23_SpecActivity_r * r23_EnzConc * (60/1000);
    r23_v_FUM_maxf = ( v_FUM + ((r23_v_FUM_maxr * mal / r23_K_mal)/(1 + (mal/r23_K_mal) + (fum/r23_K_fum))) ) * ( (1 + (mal/r23_K_mal) + (fum/r23_K_fum))/(fum/r23_K_fum) );
% Malate Dehydrogenase (MDH):
    r24_kcat_malform = 900; % [Units = s-1] Muslin et al (1995): pH 7.5 at 25oC (E.coli)
      %r24_kcat_oaaform = 21; % [Units = s-1] Muslin et al (1995): pH 7.5 at 25oC (E.coli)
      r24_MDH_EnzConc = 0.772441686009219; % Keio Multi-Omics database; Units of mg-Protein/gDCW.
      r24_MDH_EnzWeight = 64.674; % EcoCyc; Units of g/mmol.
      %r24_v_MDH_f = r24_kcat_oaaform * (3600/1000) * (r24_MDH_EnzConc/r24_MDH_EnzWeight);
      r24_v_MDH_r = r24_kcat_malform * (3600/1000) * (r24_MDH_EnzConc/r24_MDH_EnzWeight);
    %r24_v_MDH_f = r24_v_MDH_f * 96.3812845972819;
    r24_v_MDH_r = r24_v_MDH_r * 0.698918578989237;
    r24_Km_nad = 0.26 * 0.835135970965515; % Muslin et al (1995); Units of mM: pH 7.5 at 25oC (E.coli)
    r24_Km_mal = 2.6 * 0.793838713002226; % Muslin et al (1995); Units of mM: pH 7.5 at 25oC (E.coli)
    r24_Km_nadh = 0.061 * 1.88668457833965; % Muslin et al (1995); Units of mM: pH 7.5 at 25oC (E.coli)
    r24_Km_oaa = 0.049 * 1.95218504089082; % Muslin et al (1995); Units of mM: pH 7.5 at 25oC (E.coli)
    r24_v_MDH_f = (v_MDH + ( ( r24_v_MDH_r*(nadh/r24_Km_nadh)*(oaa/r24_Km_oaa) ) / ( 1 + (nadh/r24_Km_nadh) + (nadh*oaa/(r24_Km_nadh*r24_Km_oaa)) ) )) / ((nad/r24_Km_nad)*(mal/r24_Km_mal) / ( 1 + (nad/r24_Km_nad) + (nad*mal/(r24_Km_nad*r24_Km_mal)) ));
% Phosphoenolpyruvate Carboxylase (PPC):  
    r25_a = 8.2; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_b = 4.5014; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_c = 12.9; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_d = 8.0988; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_e = 0.5731; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_Km_pep = 15; % Izui et al (1981); Units of mM: pH 7.3 at 30oC.
    r25_n1 = 1.2; % Izui et al (1981); Units of mM: pH 7.3 at 30oC.
    r25_Ka_accoa = 0.001; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_Ka_fdp = 2.4540; % Estimated from Izui et al (1981) figure 1 data; Units of mM: pH 7.3 at 30oC.
    r25_Ki_mal = 2.20; % Izui et al (1983); Units of mM: pH 8.5 at 30oC.
      r25_Kpep_app1 = r25_Km_pep * (1 + (accoa/(r25_c*r25_Ka_accoa)) + (fdp/(r25_d*r25_Ka_fdp)) + (accoa*fdp*r25_e/(r25_c*r25_Ka_accoa*r25_d*r25_Ka_fdp)) ) / ( 1 + (accoa/r25_Ka_accoa) + (fdp/r25_Ka_fdp) + (accoa*fdp/(r25_Ka_accoa*r25_Ka_fdp)) ); % To account for non-essential activation by accoa and fdp to PPC rxn, as from Izui et al (1981).
      r25_Kpep_app2 = r25_Kpep_app1 * ( 1 + (mal/r25_Ki_mal) ); % Accounting for competitive inhibition of malate Vs pep, as from Izui et al (1983).
      r25_v_max_app = v_PPC / ((pep^r25_n1) / ((r25_Kpep_app2^r25_n1) + (pep^r25_n1)));
      r25_v_PPC_max = r25_v_max_app / ((1 + (r25_a*accoa/r25_Ka_accoa) + (r25_b*fdp/r25_Ka_fdp) + (r25_a*accoa*r25_b*fdp*r25_e/(r25_Ka_accoa*r25_Ka_fdp)) ) / ( 1 + (accoa/r25_Ka_accoa) + (fdp/r25_Ka_fdp) + (accoa*fdp/(r25_Ka_accoa*r25_Ka_fdp)) )); % To account for non-essential activation by accoa and fdp to PPC rxn, as from Izui et al (1981).
% Phosphoenolpyruvate Carboxykinase (PPCK):
    % Average of reported enzyme specific activities taken at different
    % dilution rates. Assumed that specific activity didn't change for
    % different dilution rates, as discussed in Yang et al (2003):
    %r26_SpecActivity = 32.3333; % Average of data of Yang et al (2003) from TableVI; Units of nmol/min/mg-Protein.
    %r26_EnzConc = 1.2753185572035; % Keio Multi-Omics database; Units of mg-Protein/gDCW.
    %r26_v_PPCK_max = r26_SpecActivity * r26_EnzConc * (60/1e6); % To get into units of mmol/(gDW*h).
    r26_Ki_atp = 0.04; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_Km_oaa = 0.67; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_Km_atp = 0.06; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_Ki_pep = 0.06; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_Ki_adp = 0.04; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_Km_pep = 0.07; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_KI_atp = 0.04; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_KI_oaa = 0.45; % Yang et al (2003): 37oC at pH 7.0 from continuous culture chemostat experiments of Yang et al (2003).
    r26_v_PPCK_max = v_PPCK / (( oaa * atp ) / ( (r26_Ki_atp*r26_Km_oaa*atp) + (r26_Km_atp*oaa) + (oaa*atp) + (r26_Ki_atp*r26_Km_oaa*pep/r26_Ki_pep) + (r26_Ki_atp*r26_Km_oaa*adp/r26_Ki_adp) + (r26_Ki_atp*r26_Km_oaa*pep*adp/(r26_Km_pep*r26_Ki_adp)) + (r26_Ki_atp*r26_Km_oaa*atp*pep/(r26_Ki_pep*r26_KI_atp)) + (r26_Ki_atp*r26_Km_oaa*oaa*adp/(r26_Ki_adp*r26_KI_oaa)) ));
% Malic Enzyme (ME1):
    r27_n1 = 1.3; % Wang et al (2006): 25oC at pH 7.2.
    r27_K_inad = 0.3317; % Value found from optimization with data of Wang et al (2006): 25oC at pH 7.2.
    r27_K_nad = 0.097; % Wang et al (2006); Value +/- 0.038mM: 25oC at pH 7.2.
    r27_K_mal = 0.420; % Wang et al (2006); Value +/- 0.174mM: 25oC at pH 7.2.
    r27_n2 = 1.1974; % Value found from optimization with data of Yamaguchi et al (1974): 30oC at pH 7.9.
    r27_L = 0.14899; % Value found from optimization with data of Yamaguchi et al (1974): 30oC at pH 7.9.
    r27_Ki = 0.016205; % Value found from optimization with data of Yamaguchi et al (1974): 30oC at pH 7.9.
    ME1_Inh = ((1 + (mal/r27_K_mal))^r27_n2) / ( (r27_L*((1 + (coa/r27_Ki))^r27_n2)) +  ((1 + (mal/r27_K_mal))^r27_n2) );
    r27_v_ME1_max = v_ME1 / (( nad * (mal^r27_n1) * ME1_Inh )/( (r27_K_inad*(r27_K_mal^r27_n1)) + (r27_K_nad*(mal^r27_n1)) + ((r27_K_mal^r27_n1)*nad) + (nad*(mal^r27_n1)) ));
% Malate Synthase (MALS):
    r28_v_MALS_max = 0;
    r28_Km_glx = 0.021; % Anstrom et al (2003): pH 8.0 at 37oC.
    r28_Ki_pyr = 1.0; % Anstrom et al (2003): pH 8.0 at 37oC.
    r28_Km_accoa = 0.009; % Anstrom et al (2003): pH 8.0 at 37oC.
% Glucose-6-Phosphate Dehydrogenase (G6PDH):
    %r29_SpecActivity = 5.7; % Taken from Sanwal et al (1970); Units of umol/min/mg-P: pH 8.0 but unknown temperature.
    %r29_EnzConc = 0.0762694433805037; % From Keio Multi-Omics Database.
    %r29_v_G6PDH_max = r29_SpecActivity * r29_EnzConc * (60/1000);
    %r29_v_G6PDH_max = r29_v_G6PDH_max * 83.9874291285696;
    r29_Ki_nadh = 0.22081 * 1.56927050864693; % Optimally estimated from Sanwal et al (1970) exp. data, from Fig7.
    r29_n = 1.7847 * 1.95139204065023; % Optimally estimated from Sanwal et al (1970) exp. data, from Fig7.
    r29_Ki_nadph_nadp = 0.01 * 1.95292558414678; % From Sanwal et al (1970): pH 7.5, but unknown temperature.
    r29_Ki_nadph_g6p = 0.69684 * 1.99925284809769; % Optimally estimated from Sanwal et al (1970) exp. data, from Fig3.
    r29_Ki_nadp = 0.07 * 0.510987838965108; % From Sanwal et al (1970): pH 7.5, but unknown temperature.
    r29_K_g6p = 0.07 * 0.502211068792437; % From Sanwal et al (1970): pH 7.5, but unknown temperature.
    r29_K_nadp = 0.015 * 0.50156715128819; % From Sanwal et al (1970): pH 7.5, but unknown temperature.
      r29_NonCompInh_byNADH = 1 / (1 + (nadh/r29_Ki_nadh)^r29_n);
      r29_CompInh_nadp = (1 + (nadph/r29_Ki_nadph_nadp));
      r29_NonCompInh_g6p = 1 / (1 + (nadph/r29_Ki_nadph_g6p));
      r29_v_G6PDH_max = v_G6PDH / ((nadp * g6p * r29_NonCompInh_g6p * r29_NonCompInh_byNADH) / ( (r29_Ki_nadp*r29_K_g6p*r29_CompInh_nadp) + (r29_K_nadp*r29_CompInh_nadp*g6p) + (nadp*r29_K_g6p) + (nadp*g6p) ));
% Phosphogluconate Dehydrogenase (GND):
    %r30_v_GND_max = 36.4398307714076; % As estimated from optimization.
    r30_K_6pgc = 0.1; % DeSilva et al (1979): 25oC at pH 7.5.
    r30_Ki_fdp = 0.025; % DeSilva et al (1979): 25oC at pH 7.5.
    r30_K_nadp = 0.028; % DeSilva et al (1979): 25oC at pH 7.5.
    r30_Ki_nadph = 0.01; % DeSilva et al (1979): 25oC at pH 7.5.
    r30_Ki_atp = 3.0; % DeSilva et al (1979): 25oC at pH 7.5.
    r30_v_GND_max = v_GND / (( m_6pgc * nadp ) / ( ( m_6pgc + (r30_K_6pgc*(1+(fdp/r30_Ki_fdp))*(1 + (atp/r30_Ki_atp))) ) * ( nadp + (r30_K_nadp*(1+(nadph/r30_Ki_nadph))) ) ));
% Ribulose 5-Phosphate 3-Epimerase (RPE):
    % r31_v_RPE_max = vmax{31};
    r31_K_eq = 1.4; % Chassagnole et al (2002)
    r31_v_RPE_max = v_RPE / (ru5pD - (xu5pD/r31_K_eq));
% Ribose 5-Phosphate Isomerase (RPI):
    % r32_v_RPI_max = vmax{32};
    r32_K_eq = 4.0; % Chassagnole et al (2002)
    r32_v_RPI_max = v_RPI / (ru5pD - (r5p/r32_K_eq));
% (TKT1):
    %r33_SpecActivity = 50.4; % From Sprenger et al (1995)
    %r33_EnzConc = 1.43863598222968; % Taken from Keio Multi-Omics Database.
    %r33_v_TKT1_max = r33_SpecActivity * r33_EnzConc * (60/1000);
    %r33_v_TKT1_max = r33_v_TKT1_max * 13.5551173607004;
    r33_K_eq = 1.2 * 0.530611658438192; % Chassagnole et al (2002)
    r33_K_r5p = 1.4 * 1.8393694588892; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r33_K_xu5pD = 0.16 * 1.79819168995143; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r33_K_g3p = 2.1 * 1.13118533448139; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r33_K_s7p = 4 * 1.30031672802347; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r33_v_TKT1_max = v_TKT1 / (((r5p*xu5pD) - ((g3p*s7p)/r33_K_eq)) / ( ((r33_K_r5p*(1+(g3p/r33_K_g3p))) + r5p) * ((r33_K_xu5pD*(1+(s7p/r33_K_s7p))) + xu5pD) ));
% (TKT2):
    %r34_SpecActivity = 50.4; % From Sprenger et al (1995)
    %r34_EnzConc = 1.43863598222968; % Taken from Keio Multi-Omics Database.
    %r34_v_TKT2_max = r34_SpecActivity * r34_EnzConc * (60/1000);
    %r34_v_TKT2_max = r34_v_TKT2_max * 2.4493913312742;
    r34_K_eq = 10.0 * 1.23100503719289; % Chassagnole et al (2002)
    r34_K_e4p = 0.09 * 0.830066048942311; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r34_K_f6p = 1.1 * 1.8584945199174; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r34_K_xu5pD = 0.16 * 1.62823937610058; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r34_K_g3p = 2.1 * 1.02836307255119; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r34_v_TKT2_max = v_TKT2 / (((e4p*xu5pD) - ((f6p*g3p)/r34_K_eq)) / ( ((r34_K_e4p*(1+(f6p/r34_K_f6p))) + e4p) * ((r34_K_xu5pD*(1+(g3p/r34_K_g3p))) + xu5pD) ));
% (TALA):
    %r35_SpecActivity = 60; % Sprenger et al (1995); Units of umol/min/mg-P: 30oC at pH 8.5.
    %r35_EnzConc = 0.673907788204659; % Keio Multi-Omics Database; Units of mg-Protein/gDCW.
    %r35_v_TALA_maxf = r35_SpecActivity * r35_EnzConc * (60/1000); % To convert the v_max to units of mmol/gDCW/h, as required.
    %r35_v_TALA_maxf = r35_v_TALA_maxf * 1.31557036349798;
    r35_K_eq = 1.05 * 1.2147571085545; % Chassagnole et al (2002)
    r35_K_g3p = 0.038 * 0.500030618743079; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r35_K_e4p = 0.09 * 1.43496014395274; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r35_K_s7p = 0.285 * 0.500202375260204; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r35_K_f6p = 1.2 * 0.957800183321665; % Sprenger et al (1995); Units of mM: 30oC at pH 8.5.
    r35_v_TALA_maxf = v_TALA / (((g3p * s7p) - (e4p * f6p / r35_K_eq)) / ( ((r35_K_g3p*(1 + (e4p/r35_K_e4p))) + g3p) * ((r35_K_s7p*(1 + (f6p/r35_K_f6p))) + s7p) ));
